<?php
/**
 * VPS服务器到期通知脚本
 * 
 * 此脚本用于检查服务器到期时间，并在服务器即将到期时发送通知
 * 可以通过cron任务每天自动运行
 */

// 设置时区
date_default_timezone_set('Asia/Shanghai');

// 数据文件路径
$dataDir = __DIR__ . '/data';
$serversFile = $dataDir . '/servers.json';
$configFile = $dataDir . '/config.json';

// 日志文件
$logFile = $dataDir . '/notification_log.txt';

// 检查数据文件是否存在
if (!file_exists($serversFile) || !file_exists($configFile)) {
    logMessage("错误：数据文件不存在");
    exit(1);
}

// 读取服务器数据
$serversData = json_decode(file_get_contents($serversFile), true);
if (!$serversData) {
    logMessage("错误：无法解析服务器数据");
    exit(1);
}

// 读取配置数据
$configData = json_decode(file_get_contents($configFile), true);
if (!$configData) {
    logMessage("错误：无法解析配置数据");
    exit(1);
}

// 当前日期
$today = date('Y-m-d');
$todayTimestamp = strtotime($today);

// 检查每个服务器
$expiringSoon = [];
foreach ($serversData as $server) {
    // 跳过已归档的服务器
    if (isset($server['renewStatus']) && $server['renewStatus'] === 'archived') {
        continue;
    }
    
    // 检查是否有到期日期
    if (!isset($server['expireDate']) || empty($server['expireDate'])) {
        continue;
    }
    
    $expireDate = $server['expireDate'];
    $expireTimestamp = strtotime($expireDate);
    
    // 如果已经过期
    if ($expireTimestamp < $todayTimestamp) {
        $daysOverdue = floor(($todayTimestamp - $expireTimestamp) / (60 * 60 * 24));
        // 只在过期后5天内提醒
        if ($daysOverdue <= 5) {
            $expiringSoon[] = [
                'server' => $server,
                'days' => -$daysOverdue,
                'status' => 'overdue'
            ];
        }
        continue;
    }
    
    // 计算剩余天数
    $daysLeft = floor(($expireTimestamp - $todayTimestamp) / (60 * 60 * 24));
    
    // 到期前10天开始每天提醒
    if ($daysLeft <= 10) {
        $expiringSoon[] = [
            'server' => $server,
            'days' => $daysLeft,
            'status' => 'expiring_soon'
        ];
    }
}

// 如果没有即将到期的服务器，退出
if (empty($expiringSoon)) {
    logMessage("信息：没有即将到期的服务器");
    exit(0);
}

// 构建通知消息
$message = "📢 VPS服务器到期提醒\n\n";

foreach ($expiringSoon as $item) {
    $server = $item['server'];
    $days = $item['days'];
    $status = $item['status'];
    
    $serverName = $server['serverName'] ?? '未命名服务器';
    $provider = $server['provider'] ?? '未知服务商';
    $expireDate = $server['expireDate'] ?? '未知日期';
    
    if ($status === 'overdue') {
        $message .= "❗ <b>{$serverName}</b> ({$provider}) 已过期 <b>" . abs($days) . "</b> 天\n";
        $message .= "   到期日期：{$expireDate}\n\n";
    } else {
        $message .= "⚠️ <b>{$serverName}</b> ({$provider}) 将在 <b>{$days}</b> 天后到期\n";
        $message .= "   到期日期：{$expireDate}\n\n";
    }
}

// 发送通知
$notificationResults = [];

// 发送Telegram通知
if (!empty($configData['telegram_bot_token']) && !empty($configData['telegram_chat_id'])) {
    $telegramResult = sendTelegramNotification(
        $configData['telegram_bot_token'],
        $configData['telegram_chat_id'],
        $message,
        $configData['telegram_proxy'] ?? ''
    );
    $notificationResults['telegram'] = $telegramResult;
}

// 发送Ntfy通知
if (!empty($configData['ntfy_server']) && !empty($configData['ntfy_topic'])) {
    $ntfyResult = sendNtfyNotification(
        $configData['ntfy_server'],
        $configData['ntfy_topic'],
        $message
    );
    $notificationResults['ntfy'] = $ntfyResult;
}

// 发送Bark通知
if (!empty($configData['bark_server']) && !empty($configData['bark_device_key'])) {
    $barkResult = sendBarkNotification(
        $configData['bark_server'],
        $configData['bark_device_key'],
        $message
    );
    $notificationResults['bark'] = $barkResult;
}

// 记录通知结果
logMessage("通知结果: " . json_encode($notificationResults, JSON_UNESCAPED_UNICODE));

/**
 * 发送Telegram通知
 */
function sendTelegramNotification($botToken, $chatId, $message, $proxy = '') {
    $url = "https://api.telegram.org/bot{$botToken}/sendMessage";
    if (!empty($proxy)) {
        $url = rtrim($proxy, '/') . "/bot{$botToken}/sendMessage";
    }
    
    $data = [
        'chat_id' => $chatId,
        'text' => $message,
        'parse_mode' => 'HTML'
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($error) {
        return ['success' => false, 'message' => '网络错误: ' . $error];
    }
    
    if ($httpCode !== 200) {
        return ['success' => false, 'message' => 'HTTP错误: ' . $httpCode];
    }
    
    $result = json_decode($response, true);
    if ($result && $result['ok']) {
        return ['success' => true, 'message' => '通知发送成功'];
    } else {
        $errorMsg = $result['description'] ?? '未知错误';
        return ['success' => false, 'message' => 'Telegram API错误: ' . $errorMsg];
    }
}

/**
 * 发送Ntfy通知
 */
function sendNtfyNotification($server, $topic, $message) {
    // 去除HTML标签，Ntfy不支持
    $plainMessage = strip_tags($message);
    
    $url = rtrim($server, '/') . '/' . $topic;
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $plainMessage);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Title: VPS服务器到期提醒',
        'Priority: high',
        'Tags: warning'
    ]);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($error) {
        return ['success' => false, 'message' => '网络错误: ' . $error];
    }
    
    if ($httpCode === 200) {
        return ['success' => true, 'message' => '通知发送成功'];
    } else {
        return ['success' => false, 'message' => 'HTTP错误: ' . $httpCode];
    }
}

/**
 * 发送Bark通知
 */
function sendBarkNotification($server, $deviceKey, $message) {
    // 去除HTML标签，Bark不支持
    $plainMessage = strip_tags($message);
    
    $title = urlencode('VPS服务器到期提醒');
    $content = urlencode($plainMessage);
    $url = rtrim($server, '/') . '/' . $deviceKey . '/' . $title . '/' . $content;
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($error) {
        return ['success' => false, 'message' => '网络错误: ' . $error];
    }
    
    if ($httpCode === 200) {
        $result = json_decode($response, true);
        if ($result && $result['code'] === 200) {
            return ['success' => true, 'message' => '通知发送成功'];
        } else {
            $errorMsg = $result['message'] ?? '未知错误';
            return ['success' => false, 'message' => 'Bark API错误: ' . $errorMsg];
        }
    } else {
        return ['success' => false, 'message' => 'HTTP错误: ' . $httpCode];
    }
}

/**
 * 记录日志
 */
function logMessage($message) {
    global $logFile;
    $timestamp = date('Y-m-d H:i:s');
    $logEntry = "[{$timestamp}] {$message}" . PHP_EOL;
    
    // 确保日志目录存在
    $logDir = dirname($logFile);
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    
    file_put_contents($logFile, $logEntry, FILE_APPEND);
    echo $logEntry; // 同时输出到控制台
}