// 分类管理相关功能
let categories = [];

// 页面加载完成后初始化
document.addEventListener('DOMContentLoaded', function() {
    loadCategories();
    
    // 绑定表单提交事件
    document.getElementById('categoryForm').addEventListener('submit', function(e) {
        e.preventDefault();
        saveCategory();
    });
    
    // 绑定颜色选择器事件
    const colorPicker = document.getElementById('categoryColor');
    const colorText = document.getElementById('categoryColorText');
    
    colorPicker.addEventListener('change', function() {
        colorText.value = this.value;
    });
    
    colorText.addEventListener('input', function() {
        if (/^#[0-9A-F]{6}$/i.test(this.value)) {
            colorPicker.value = this.value;
        }
    });
});

// 加载分类列表
function loadCategories() {
    fetch('api.php?action=getCategories')
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                categories = data.categories;
                renderCategoryList();
            } else {
                showToast('加载分类列表失败', 'error');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showToast('网络错误', 'error');
        });
}

// 渲染分类列表
function renderCategoryList() {
    const listContainer = document.getElementById('categoryList');
    
    if (categories.length === 0) {
        listContainer.innerHTML = `
            <div class="text-center py-4">
                <i class="bi bi-tags text-muted" style="font-size: 3rem;"></i>
                <p class="text-muted mt-2">暂无分类，点击上方按钮添加第一个分类</p>
            </div>
        `;
        return;
    }
    
    let html = `
        <div class="table-responsive">
            <table class="table table-hover">
                <thead>
                    <tr>
                        <th>分类名称</th>
                        <th>颜色预览</th>
                        <th>描述</th>
                        <th>服务器数量</th>
                        <th>创建时间</th>
                        <th>操作</th>
                    </tr>
                </thead>
                <tbody>
    `;
    
    categories.forEach(category => {
        html += `
            <tr>
                <td><strong>${escapeHtml(category.name)}</strong></td>
                <td>
                    <span class="badge" style="background-color: ${escapeHtml(category.color)}; color: ${getContrastColor(category.color)}">
                        ${escapeHtml(category.name)}
                    </span>
                </td>
                <td>${category.description ? escapeHtml(category.description) : '<span class="text-muted">无</span>'}</td>
                <td>
                    <span class="badge bg-secondary">${category.server_count || 0}</span>
                </td>
                <td>${formatDateTime(category.created_at)}</td>
                <td>
                    <button class="btn btn-sm btn-outline-primary me-1" onclick="editCategory(${category.id})">
                        <i class="bi bi-pencil"></i> 编辑
                    </button>
                    <button class="btn btn-sm btn-outline-danger" onclick="deleteCategory(${category.id}, '${escapeHtml(category.name)}', ${category.server_count || 0})">
                        <i class="bi bi-trash"></i> 删除
                    </button>
                </td>
            </tr>
        `;
    });
    
    html += `
                </tbody>
            </table>
        </div>
    `;
    
    listContainer.innerHTML = html;
}

// 显示添加分类模态框
function showAddCategoryModal() {
    document.getElementById('categoryModalTitle').textContent = '添加分类';
    document.getElementById('categoryForm').reset();
    document.getElementById('categoryId').value = '';
    document.getElementById('categoryColor').value = '#0d6efd';
    document.getElementById('categoryColorText').value = '#0d6efd';
    
    const modal = new bootstrap.Modal(document.getElementById('categoryModal'));
    modal.show();
}

// 编辑分类
function editCategory(id) {
    const category = categories.find(c => c.id === id);
    if (!category) return;
    
    document.getElementById('categoryModalTitle').textContent = '编辑分类';
    document.getElementById('categoryId').value = category.id;
    document.getElementById('categoryName').value = category.name;
    document.getElementById('categoryColor').value = category.color;
    document.getElementById('categoryColorText').value = category.color;
    document.getElementById('categoryDescription').value = category.description || '';
    
    const modal = new bootstrap.Modal(document.getElementById('categoryModal'));
    modal.show();
}

// 保存分类
function saveCategory() {
    const formData = new FormData(document.getElementById('categoryForm'));
    const categoryId = formData.get('categoryId');
    
    const action = categoryId ? 'updateCategory' : 'addCategory';
    const data = {
        id: categoryId || undefined,
        name: formData.get('categoryName'),
        color: formData.get('categoryColor'),
        description: formData.get('categoryDescription')
    };
    
    console.log('Saving category:', data);
    
    fetch(`api.php?action=${action}`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify(data)
    })
    .then(response => {
        console.log('Response status:', response.status);
        return response.json();
    })
    .then(data => {
        console.log('Response data:', data);
        if (data.success) {
            showToast(categoryId ? '分类更新成功' : '分类添加成功', 'success');
            bootstrap.Modal.getInstance(document.getElementById('categoryModal')).hide();
            loadCategories();
        } else {
            showToast(data.message || '操作失败', 'error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showToast('网络错误: ' + error.message, 'error');
    });
}

// 删除分类
function deleteCategory(id, name, serverCount) {
    let confirmMessage = `确定要删除分类 "${name}" 吗？`;
    if (serverCount > 0) {
        confirmMessage += `\n\n注意：该分类下还有 ${serverCount} 个服务器，删除后这些服务器将显示为"未分类"。`;
    }
    
    if (!confirm(confirmMessage)) {
        return;
    }
    
    // 防止重复点击
    const deleteButton = event.target.closest('button');
    if (deleteButton) {
        deleteButton.disabled = true;
    }
    
    fetch('api.php?action=deleteCategory', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({
            id: id
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showToast('分类删除成功', 'success');
            loadCategories();
        } else {
            showToast(data.message || '删除失败', 'error');
            // 重新启用按钮
            if (deleteButton) {
                deleteButton.disabled = false;
            }
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showToast('网络错误', 'error');
        // 重新启用按钮
        if (deleteButton) {
            deleteButton.disabled = false;
        }
    });
}

// 工具函数
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

function formatDateTime(dateString) {
    const date = new Date(dateString);
    return date.toLocaleString('zh-CN', {
        year: 'numeric',
        month: '2-digit',
        day: '2-digit',
        hour: '2-digit',
        minute: '2-digit'
    });
}

// 根据背景色计算对比色
function getContrastColor(hexColor) {
    // 始终返回白色文字
    return '#ffffff';
}

// Toast 提示函数
function showToast(message, type = 'info') {
    // 创建 toast 容器（如果不存在）
    let toastContainer = document.getElementById('toast-container');
    if (!toastContainer) {
        toastContainer = document.createElement('div');
        toastContainer.id = 'toast-container';
        toastContainer.className = 'toast-container position-fixed top-0 end-0 p-3';
        toastContainer.style.zIndex = '9999';
        document.body.appendChild(toastContainer);
    }
    
    // 创建 toast 元素
    const toastId = 'toast-' + Date.now();
    const iconClass = type === 'success' ? 'bi-check-circle-fill text-success' : 
                     type === 'error' ? 'bi-exclamation-triangle-fill text-danger' : 
                     'bi-info-circle-fill text-primary';
    
    const toastHtml = `
        <div id="${toastId}" class="toast" role="alert">
            <div class="toast-header">
                <i class="bi ${iconClass} me-2"></i>
                <strong class="me-auto">${type === 'success' ? '成功' : type === 'error' ? '错误' : '提示'}</strong>
                <button type="button" class="btn-close" data-bs-dismiss="toast"></button>
            </div>
            <div class="toast-body">
                ${message}
            </div>
        </div>
    `;
    
    toastContainer.insertAdjacentHTML('beforeend', toastHtml);
    
    // 显示 toast
    const toastElement = document.getElementById(toastId);
    const toast = new bootstrap.Toast(toastElement, {
        autohide: true,
        delay: 3000
    });
    
    toast.show();
    
    // 自动清理
    toastElement.addEventListener('hidden.bs.toast', function() {
        toastElement.remove();
    });
}