// VPS管理系统前端脚本
class VPSManager {
    constructor() {
        this.servers = [];
        this.exchangeRates = {};
        this.currentEditingId = null;
        this.init();
    }

    init() {
        this.loadCategories(); // 先加载分类数据
        this.loadServers();
        this.bindEvents();
        this.loadExchangeRates();
        this.loadCategoryFilter();
    }

    bindEvents() {
        // 添加服务器按钮事件
        document.getElementById('addServerBtn').addEventListener('click', () => {
            this.showAddModal();
        });

        // 表单提交事件
        document.getElementById('serverForm').addEventListener('submit', (e) => {
            e.preventDefault();
            this.saveServer();
        });

        // 模态框关闭时重置表单
        const modal = document.getElementById('serverModal');
        modal.addEventListener('hidden.bs.modal', () => {
            this.resetForm();
        });

        // 自动续期开关事件
        document.getElementById('autoRenew').addEventListener('change', (e) => {
            const renewCycleGroup = document.getElementById('renewCycleGroup');
            if (e.target.checked) {
                renewCycleGroup.style.display = 'block';
            } else {
                renewCycleGroup.style.display = 'none';
            }
        });
    }

    async loadServers() {
        try {
            this.showLoading();
            const response = await fetch('api.php?action=list');
            const result = await response.json();
            
            if (result.success) {
                this.servers = result.data;
                this.exchangeRates = result.rates;
                this.renderServerList();
                this.updateStats();
            } else {
                this.showAlert('加载服务器列表失败', 'danger');
            }
        } catch (error) {
            console.error('Error loading servers:', error);
            this.showAlert('网络错误，请稍后重试', 'danger');
        } finally {
            this.hideLoading();
        }
    }

    async loadExchangeRates() {
        try {
            const response = await fetch('api.php?action=rates');
            const result = await response.json();
            
            if (result.success) {
                this.exchangeRates = result.data;
            }
        } catch (error) {
            console.error('Error loading exchange rates:', error);
        }
    }

    async loadProviders(callback) {
        try {
            const response = await fetch('api.php?action=getProviders');
            const data = await response.json();
            if (data.success) {
                const providerSelect = document.getElementById('provider');
                providerSelect.innerHTML = '<option value="">请选择服务商</option>';
                
                data.providers.forEach(provider => {
                    const option = document.createElement('option');
                    option.value = provider.name;
                    option.textContent = provider.name;
                    providerSelect.appendChild(option);
                });
                
                if (callback) callback();
            }
        } catch (error) {
            console.error('加载服务商失败:', error);
        }
    }

    async loadCategories(callback) {
        try {
            const response = await fetch('api.php?action=getCategories');
            const data = await response.json();
            if (data.success) {
                // 保存分类数据到实例属性中，供其他方法使用
                this.categories = data.categories;
                
                const categoriesContainer = document.getElementById('categoriesCheckboxes');
                categoriesContainer.innerHTML = '';
                
                data.categories.forEach(category => {
                    const checkboxDiv = document.createElement('div');
                    checkboxDiv.className = 'form-check';
                    
                    const checkbox = document.createElement('input');
                    checkbox.type = 'checkbox';
                    checkbox.className = 'form-check-input';
                    checkbox.id = `category_${category.id}`;
                    checkbox.value = category.name;
                    checkbox.name = 'categories[]';
                    
                    const label = document.createElement('label');
                    label.className = 'form-check-label';
                    label.htmlFor = `category_${category.id}`;
                    label.innerHTML = `<span class="badge me-1" style="background-color: ${category.color}">&nbsp;</span>${category.name}`;
                    
                    checkboxDiv.appendChild(checkbox);
                    checkboxDiv.appendChild(label);
                    categoriesContainer.appendChild(checkboxDiv);
                });
                
                if (callback) callback();
            }
        } catch (error) {
            console.error('加载分类失败:', error);
        }
    }

    async loadCategoryFilter() {
        try {
            const response = await fetch('api.php?action=getCategories');
            const data = await response.json();
            if (data.success) {
                const categoryFilter = document.getElementById('categoryFilter');
                if (categoryFilter) {
                    // 清空现有选项，保留"所有分类"
                    categoryFilter.innerHTML = '<option value="">所有分类</option>';
                    
                    // 添加分类选项
                    data.categories.forEach(category => {
                        const option = document.createElement('option');
                        option.value = category.name;
                        option.textContent = category.name;
                        categoryFilter.appendChild(option);
                    });
                }
            }
        } catch (error) {
            console.error('加载分类筛选选项失败:', error);
        }
    }

    renderServerList() {
        const container = document.getElementById('serverList');
        if (!container) return;

        // 过滤服务器
        let filteredServers = this.servers;
        
        // 状态过滤 - 默认只显示活跃服务器
        const statusFilter = document.getElementById('statusFilter');
        if (statusFilter && statusFilter.value === 'archived') {
            // 只有明确选择归档时才显示归档服务器
            filteredServers = filteredServers.filter(server => server.renewStatus === 'archived');
        } else {
            // 默认只显示活跃服务器（不显示归档服务器）
            filteredServers = filteredServers.filter(server => server.renewStatus !== 'archived');
        }
        
        // 分类过滤
        const categoryFilter = document.getElementById('categoryFilter');
        if (categoryFilter && categoryFilter.value) {
            filteredServers = filteredServers.filter(server => {
                // 支持新的categories数组和旧的category字段
                if (server.categories && Array.isArray(server.categories)) {
                    return server.categories.includes(categoryFilter.value);
                }
                return server.category === categoryFilter.value;
            });
        }

        if (filteredServers.length === 0) {
            // 检查是否是归档页面
            const statusFilter = document.getElementById('statusFilter');
            const isArchivePage = statusFilter && statusFilter.value === 'archived';
            
            if (isArchivePage) {
                container.innerHTML = `
                    <div class="empty-state">
                        <div class="empty-icon">📦</div>
                        <h5>暂无归档服务器</h5>
                        <p>当前没有已归档的服务器</p>
                        <small class="text-muted">归档的服务器将在这里显示，方便您查看历史记录</small>
                    </div>
                `;
            } else {
                container.innerHTML = `
                    <div class="empty-state">
                        <div class="empty-icon">🖥️</div>
                        <h5>暂无服务器</h5>
                        <p>点击上方按钮添加您的第一台服务器</p>
                    </div>
                `;
            }
            return;
        }
        
        // 开始行容器
        let serversHtml = '<div class="row">';
        
        // 映射服务器卡片
        serversHtml += filteredServers.map(server => {
            const daysRemaining = Math.ceil((new Date(server.expireDate) - new Date()) / (1000 * 60 * 60 * 24));
            let statusBadge = '';
            
            if (server.renewStatus === 'archived') {
                statusBadge = '<span class="badge bg-secondary">已归档</span>';
            } else {
                statusBadge = this.getStatusBadge(daysRemaining);
            }
            
            // 分类标签 - 支持多个分类，使用动态颜色
            let categoryBadges = '';
            const serverCategories = server.categories || (server.category ? [server.category] : ['其它']);
            
            serverCategories.forEach(category => {
                // 从categories数组中查找对应的颜色
                const categoryData = this.categories ? this.categories.find(cat => cat.name === category) : null;
                const categoryColor = categoryData ? categoryData.color : '#6c757d';
                categoryBadges += `<span class="badge me-1" style="background-color: ${categoryColor}; color: #ffffff;">${category}</span>`;
            });
            
            // 操作按钮
            let actionButtons = '';
            if (server.renewStatus === 'archived') {
                actionButtons = '<span class="text-muted">已归档，不可操作</span>';
            } else {
                actionButtons = `
                    <button type="button" class="btn btn-sm btn-primary" onclick="vpsManager.editServer('${server.id}')">
                        编辑
                    </button>
                    <button type="button" class="btn btn-sm btn-success" onclick="vpsManager.renewServer('${server.id}')">
                        续期
                    </button>
                    <button type="button" class="btn btn-sm btn-warning" onclick="vpsManager.archiveServer('${server.id}')">
                        归档
                    </button>
                    ${server.notes && server.notes.startsWith('http') ? 
                        `<a href="${server.notes}" target="_blank" class="btn btn-sm btn-info">
                            访问
                        </a>` : ''}
                `;
            }
            
            // 面板账号密码显示
            const panelAccountHtml = server.panelAccount ? `<div class="server-panel"><strong>账号:</strong> ${this.escapeHtml(server.panelAccount)}</div>` : '';
            const panelPasswordHtml = server.panelPassword ? `
                <div class="server-panel">
                    <strong>密码:</strong> 
                    <span class="password-field">••••••••</span>
                    <button class="btn btn-sm btn-outline-secondary ms-1" onclick="copyPassword('panel-password-${server.id}')"><i class="bi bi-clipboard"></i></button>
                    <input type="hidden" id="panel-password-${server.id}" value="${this.escapeHtml(server.panelPassword)}">
                </div>` : '';
            
            return `
                <div class="col-sm-6 col-md-4 col-lg-3 mb-3">
                    <div class="card server-card h-100 ${server.renewStatus === 'archived' ? 'archived' : ''}">
                        <div class="card-header d-flex justify-content-between align-items-center py-2">
                            <h6 class="mb-0">${this.escapeHtml(server.serverName)}</h6>
                            <div>
                                ${categoryBadges}
                                <small class="text-muted ms-2">${this.escapeHtml(server.provider)}</small>
                            </div>
                        </div>
                        <div class="card-body">
                            <div class="server-cost">
                                <span class="cost-amount">¥${(server.cost_cny || 0).toFixed(2)}</span>
                                <small class="text-muted">(${server.cost} ${server.currency})</small>
                            </div>
                        
                            <div class="server-dates">
                                <span>入手: ${server.purchaseDate}</span> | 
                                <span>到期: ${server.expireDate}</span> | 
                                ${statusBadge}
                                ${server.autoRenew ? '<span class="badge bg-info ms-1">🔄 自动续期</span>' : ''}
                            </div>
                            
                            ${panelAccountHtml}
                            ${panelPasswordHtml}
                            ${server.notes ? `<div class="server-notes">${this.escapeHtml(server.notes)}</div>` : ''}
                            
                            <div class="server-actions">
                                ${actionButtons}
                            </div>
                        </div>
                    </div>
                </div>
            `;
        }).join('');
        
        // 闭合行容器
        serversHtml += '</div>';

        container.innerHTML = serversHtml;
    }

    getStatusBadge(daysRemaining) {
        if (daysRemaining < 0) {
            return `<span class="badge bg-danger">已过期 ${Math.abs(daysRemaining)} 天</span>`;
        } else if (daysRemaining <= 7) {
            return `<span class="badge bg-warning">还有 ${daysRemaining} 天到期</span>`;
        } else if (daysRemaining <= 30) {
            return `<span class="badge bg-warning">还有 ${daysRemaining} 天到期</span>`;
        } else {
            return `<span class="badge bg-success">还有 ${daysRemaining} 天到期</span>`;
        }
    }

    updateStats() {
        const totalServers = this.servers.length;
        const activeServers = this.servers.filter(s => s.renewStatus !== 'archived').length;
        const archivedServers = this.servers.filter(s => s.renewStatus === 'archived').length;
        const totalCostCny = this.servers.reduce((sum, server) => sum + (server.cost_cny || 0), 0);
        const expiringSoon = this.servers.filter(server => {
            const days = Math.ceil((new Date(server.expireDate) - new Date()) / (1000 * 60 * 60 * 24));
            return days <= 30 && days >= 0 && server.renewStatus !== 'archived';
        }).length;
        const expired = this.servers.filter(server => {
            const days = Math.ceil((new Date(server.expireDate) - new Date()) / (1000 * 60 * 60 * 24));
            return days < 0 && server.renewStatus !== 'archived';
        }).length;

        // 如果页面有统计区域，更新统计信息
        const statsContainer = document.getElementById('statsContainer');
        if (statsContainer) {
            statsContainer.innerHTML = `
                <div class="row">
                    <div class="col-md-3">
                        <div class="card stats-card">
                            <div class="stats-number">${activeServers}</div>
                            <div class="stats-label">活跃服务器</div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card stats-card">
                            <div class="stats-number">${archivedServers}</div>
                            <div class="stats-label">已归档</div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card stats-card">
                            <div class="stats-number text-warning">${expiringSoon}</div>
                            <div class="stats-label">即将到期</div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card stats-card">
                            <div class="stats-number text-danger">${expired}</div>
                            <div class="stats-label">已过期</div>
                        </div>
                    </div>
                </div>
            `;
        }
    }

    showAddModal() {
        this.currentEditingId = null;
        this.resetForm();
        this.loadProviders();
        this.loadCategories();
        document.getElementById('modalTitle').textContent = '添加服务器';
        const modal = new bootstrap.Modal(document.getElementById('serverModal'));
        modal.show();
    }

    editServer(serverId) {
        const server = this.servers.find(s => s.id === serverId);
        if (!server) return;
        
        // 检查是否已归档
        if (server.renewStatus === 'archived') {
            this.showToast('📦 已归档的服务器不允许修改，如需修改请先取消归档状态', 'warning');
            return;
        }

        this.currentEditingId = serverId;
        
        // 先加载选项
        this.loadProviders(() => {
            document.getElementById('provider').value = server.provider;
        });
        
        this.loadCategories(() => {
            // 处理多选分类
            let serverCategories = [];
            if (server.categories) {
                serverCategories = Array.isArray(server.categories) ? server.categories : [server.categories];
            } else if (server.category) {
                serverCategories = [server.category];
            }
            
            // 设置选中的分类复选框
            const categoryCheckboxes = document.querySelectorAll('input[name="categories[]"]');
            categoryCheckboxes.forEach(checkbox => {
                checkbox.checked = serverCategories.includes(checkbox.value);
            });
        });
        
        // 填充表单
        document.getElementById('serverId').value = server.id;
        document.getElementById('serverName').value = server.serverName;
        document.getElementById('cost').value = server.cost;
        document.getElementById('currency').value = server.currency;
        document.getElementById('purchaseDate').value = server.purchaseDate;
        document.getElementById('expireDate').value = server.expireDate;
        document.getElementById('renewStatus').value = server.renewStatus || 'active';
        document.getElementById('autoRenew').checked = server.autoRenew || false;
        document.getElementById('renewCycle').value = server.renewCycle || '30';
        document.getElementById('notes').value = server.notes || '';
        
        // 根据自动续期状态显示/隐藏续费周期选择
        const renewCycleGroup = document.getElementById('renewCycleGroup');
        if (server.autoRenew) {
            renewCycleGroup.style.display = 'block';
        } else {
            renewCycleGroup.style.display = 'none';
        }
        
        document.getElementById('modalTitle').textContent = '编辑服务器';
        const modal = new bootstrap.Modal(document.getElementById('serverModal'));
        modal.show();
    }

    async saveServer() {
        const form = document.getElementById('serverForm');
        const formData = new FormData(form);
        const action = this.currentEditingId ? 'update' : 'add';
        formData.append('action', action);
        
        // 处理多选分类数据
        const categoryCheckboxes = document.querySelectorAll('input[name="categories[]"]:checked');
        const selectedCategories = Array.from(categoryCheckboxes).map(checkbox => checkbox.value);
        
        // 如果没有选中任何分类，使用默认分类
        if (selectedCategories.length === 0) {
            selectedCategories.push('其它');
        }
        
        // 删除所有categories相关的数据，重新添加
        formData.delete('categories');
        formData.delete('categories[]');
        formData.append('categories', JSON.stringify(selectedCategories));

        try {
            const response = await fetch('api.php', {
                method: 'POST',
                body: formData
            });
            
            const result = await response.json();
            
            if (result.success) {
                const action = this.currentEditingId ? '更新' : '添加';
                this.showToast(`✅ 服务器${action}成功！信息已保存到系统中`, 'success');
                const modal = bootstrap.Modal.getInstance(document.getElementById('serverModal'));
                modal.hide();
                await this.loadServers();
            } else {
                this.showToast(result.message || '❌ 操作失败，请检查输入信息后重试', 'danger');
            }
        } catch (error) {
            console.error('Error saving server:', error);
            this.showToast('💥 网络错误，保存失败，请检查网络连接后重试', 'danger');
        }
    }

    async deleteServer(serverId) {
        const server = this.servers.find(s => s.id === serverId);
        if (!server) return;
        
        // 检查是否已归档
        if (server.renewStatus === 'archived') {
            this.showToast('📦 已归档的服务器不允许删除，如需删除请先取消归档状态', 'warning');
            return;
        }

        const confirmed = await this.showConfirm(
            '删除服务器',
            `确定要删除服务器 "${server.serverName}" 吗？此操作不可撤销。`,
            'danger'
        );
        
        if (!confirmed) return;

        try {
            const formData = new FormData();
            formData.append('action', 'delete');
            formData.append('serverId', serverId);

            const response = await fetch('api.php', {
                method: 'POST',
                body: formData
            });
            
            const result = await response.json();
            
            if (result.success) {
                this.showToast(`🗑️ 服务器 "${server.serverName}" 已成功删除`, 'success');
                await this.loadServers();
            } else {
                this.showToast(result.message || '❌ 删除失败，请稍后重试', 'danger');
            }
        } catch (error) {
            console.error('Error deleting server:', error);
            this.showToast('💥 网络错误，删除失败，请检查网络连接后重试', 'danger');
        }
    }
    
    async archiveServer(serverId) {
        const server = this.servers.find(s => s.id === serverId);
        if (!server) return;

        const confirmed = await this.showConfirm(
            '归档服务器',
            `确定要归档服务器 "${server.serverName}" 吗？归档后将无法修改或删除，但可以查看历史记录。`,
            'warning'
        );
        
        if (!confirmed) return;

        try {
            this.showLoading();
            const formData = new FormData();
            formData.append('action', 'archive');
            formData.append('serverId', serverId);

            const response = await fetch('api.php', {
                method: 'POST',
                body: formData
            });
            
            const result = await response.json();
            this.hideLoading();
            
            if (result.success) {
                this.showToast(`📦 服务器 "${server.serverName}" 已成功归档，可在归档列表中查看`, 'success');
                await this.loadServers();
            } else {
                this.showToast(result.message || '❌ 归档失败，请稍后重试', 'danger');
            }
        } catch (error) {
            this.hideLoading();
            console.error('Error archiving server:', error);
            this.showToast('💥 网络错误，归档失败，请检查网络连接后重试', 'danger');
        }
    }
    
    async renewServer(id) {
        const server = this.servers.find(s => s.id === id);
        if (!server) return;
        
        if (server.renewStatus === 'archived') {
            this.showToast('📦 已归档的服务器不允许续期，如需续期请先取消归档状态', 'warning');
            return;
        }
        
        const confirmed = await this.showConfirm(
            '续期服务器',
            `确定要为服务器 "${server.serverName}" 续期30天吗？到期时间将延长至下个月。`,
            'info'
        );
        
        if (!confirmed) return;
        
        try {
            this.showLoading();
            const formData = new FormData();
            formData.append('action', 'renew');
            formData.append('serverId', id);
            
            const response = await fetch('api.php', {
                method: 'POST',
                body: formData
            });
            
            const data = await response.json();
            this.hideLoading();
            
            if (data.success) {
                this.showToast(`🎉 续期成功！服务器 "${server.serverName}" 已延长30天，新的到期时间请查看服务器列表。`, 'success');
                await this.loadServers();
            } else {
                this.showToast(data.message || '❌ 续期失败，请稍后重试或联系管理员', 'danger');
            }
        } catch (error) {
            this.hideLoading();
            this.showToast('💥 网络错误，续期失败: ' + error.message, 'danger');
        }
    }

    resetForm() {
        document.getElementById('serverForm').reset();
        this.currentEditingId = null;
        
        // 重置时隐藏续费周期选择
        document.getElementById('renewCycleGroup').style.display = 'none';
    }

    showAlert(message, type = 'info') {
        this.showToast(message, type);
    }

    showToast(message, type = 'info') {
        const typeConfig = {
            success: { icon: '✓', title: '成功' },
            danger: { icon: '✕', title: '错误' },
            warning: { icon: '!', title: '警告' },
            info: { icon: 'i', title: '提示' }
        };

        const config = typeConfig[type] || typeConfig.info;
        
        const toastHtml = `
            <div class="custom-toast ${type}">
                <div class="toast-header">
                    <div class="toast-icon ${type}">${config.icon}</div>
                    <div class="toast-title">${config.title}</div>
                    <button class="toast-close" onclick="this.parentElement.parentElement.remove()">&times;</button>
                </div>
                <div class="toast-body">
                    ${this.escapeHtml(message)}
                </div>
            </div>
        `;
        
        const toastDiv = document.createElement('div');
        toastDiv.innerHTML = toastHtml;
        document.body.appendChild(toastDiv.firstElementChild);
        
        // 4秒后自动消失
        setTimeout(() => {
            const toast = document.querySelector('.custom-toast');
            if (toast) {
                toast.style.animation = 'slideOutRight 0.3s ease-out';
                setTimeout(() => toast.remove(), 300);
            }
        }, 4000);
    }

    showConfirm(title, message, type = 'warning') {
        return new Promise((resolve) => {
            const typeConfig = {
                warning: { icon: '!', btnClass: 'danger' },
                danger: { icon: '✕', btnClass: 'danger' },
                info: { icon: 'i', btnClass: 'primary' }
            };

            const config = typeConfig[type] || typeConfig.warning;
            
            const modalHtml = `
                <div class="custom-modal-overlay">
                    <div class="custom-modal">
                        <div class="custom-modal-header">
                            <div class="custom-modal-icon ${type}">${config.icon}</div>
                            <div class="custom-modal-title">${title}</div>
                        </div>
                        <div class="custom-modal-body">
                            ${this.escapeHtml(message)}
                        </div>
                        <div class="custom-modal-footer">
                            <button class="custom-modal-btn secondary" onclick="this.closest('.custom-modal-overlay').remove(); window.currentConfirmResolve(false);">取消</button>
                            <button class="custom-modal-btn ${config.btnClass}" onclick="this.closest('.custom-modal-overlay').remove(); window.currentConfirmResolve(true);">确认</button>
                        </div>
                    </div>
                </div>
            `;
            
            window.currentConfirmResolve = resolve;
            
            const modalDiv = document.createElement('div');
            modalDiv.innerHTML = modalHtml;
            document.body.appendChild(modalDiv.firstElementChild);
        });
    }

    showLoading() {
        const container = document.getElementById('serverList');
        container.innerHTML = `
            <div class="loading">
                <div class="spinner-border" role="status">
                    <span class="visually-hidden">加载中...</span>
                </div>
            </div>
        `;
    }

    hideLoading() {
        // Loading will be hidden when renderServerList is called
    }

    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    // 检查自动续期
    async checkAutoRenew() {
        try {
            const response = await fetch('api.php?action=auto_renew_check', {
                method: 'GET'
            });
            
            const result = await response.json();
            
            if (result.success && result.renewed_count > 0) {
                this.showToast(`🔄 自动续期完成！共有 ${result.renewed_count} 台服务器已自动续期`, 'success');
                // 重新加载服务器列表以显示更新后的到期时间
                await this.loadServers();
            }
        } catch (error) {
            console.error('自动续期检查失败:', error);
        }
    }
}

// 全局函数
function showAddModal() {
    vpsManager.showAddModal();
}

function filterByCategory(refresh = false) {
    vpsManager.renderServerList();
}

function filterByStatus(refresh = false) {
    vpsManager.renderServerList();
}

function showArchivedServers() {
    document.getElementById('statusFilter').value = 'archived';
    vpsManager.renderServerList();
}

let sortOrder = 'asc'; // 排序状态：'asc' 升序（由近到远），'desc' 降序（由远到近）

function sortByExpiry() {
    if (vpsManager && vpsManager.servers) {
        // 切换排序状态
        sortOrder = sortOrder === 'asc' ? 'desc' : 'asc';
        
        vpsManager.servers.sort((a, b) => {
            const dateA = new Date(a.expireDate);
            const dateB = new Date(b.expireDate);
            return sortOrder === 'asc' ? dateA - dateB : dateB - dateA;
        });
        
        // 更新按钮显示状态
        const sortBtn = document.querySelector('[onclick="sortByExpiry()"]');
        if (sortBtn) {
            const icon = sortBtn.querySelector('i');
            if (sortOrder === 'asc') {
                icon.className = 'bi bi-sort-up';
                sortBtn.title = '按到期时间排序（由近到远）';
            } else {
                icon.className = 'bi bi-sort-down';
                sortBtn.title = '按到期时间排序（由远到近）';
            }
        }
        
        vpsManager.renderServerList();
    }
}

// 密码显示/隐藏功能
function togglePasswordVisibility(inputId) {
    const input = document.getElementById(inputId);
    const icon = event.currentTarget.querySelector('i');
    
    if (input.type === 'password') {
        input.type = 'text';
        icon.classList.remove('bi-eye');
        icon.classList.add('bi-eye-slash');
    } else {
        input.type = 'password';
        icon.classList.remove('bi-eye-slash');
        icon.classList.add('bi-eye');
    }
}

// 复制密码功能
function copyPassword(inputId) {
    const input = document.getElementById(inputId);
    const originalType = input.type;
    
    // 临时改为文本以便复制
    input.type = 'text';
    input.select();
    document.execCommand('copy');
    
    // 恢复原始类型
    input.type = originalType;
    
    // 取消选择
    input.setSelectionRange(0, 0);
    input.blur();
    
    // 显示提示
    const button = event.currentTarget;
    const originalHTML = button.innerHTML;
    button.innerHTML = '<i class="bi bi-check"></i>';
    setTimeout(() => {
        button.innerHTML = originalHTML;
    }, 1000);
}

// 通用复制到剪贴板函数
function copyToClipboard(text, buttonElement) {
    const tempInput = document.createElement('input');
    tempInput.style.position = 'absolute';
    tempInput.style.left = '-9999px';
    tempInput.value = text;
    document.body.appendChild(tempInput);
    tempInput.select();
    document.execCommand('copy');
    document.body.removeChild(tempInput);
    
    // 显示复制成功提示
    const originalHTML = buttonElement.innerHTML;
    buttonElement.innerHTML = '<i class="bi bi-check"></i>';
    setTimeout(() => {
        buttonElement.innerHTML = originalHTML;
    }, 1000);
}

// 初始化应用
let vpsManager;
document.addEventListener('DOMContentLoaded', function() {
    vpsManager = new VPSManager();
    
    // 页面加载时检查一次自动续期
    setTimeout(() => {
        vpsManager.checkAutoRenew();
    }, 2000);
});

// 每小时更新一次汇率和检查自动续期
setInterval(() => {
    if (vpsManager) {
        vpsManager.loadExchangeRates();
        vpsManager.checkAutoRenew();
    }
}, 3600000);

// 每天凌晨2点额外检查一次自动续期
setInterval(() => {
    const now = new Date();
    if (now.getHours() === 2 && now.getMinutes() === 0 && vpsManager) {
        vpsManager.checkAutoRenew();
    }
}, 60000);