// 服务商管理相关功能
let providers = [];

// 页面加载完成后初始化
document.addEventListener('DOMContentLoaded', function() {
    loadProviders();
    
    // 绑定表单提交事件
    document.getElementById('providerForm').addEventListener('submit', function(e) {
        e.preventDefault();
        saveProvider();
    });
});

// 加载服务商列表
function loadProviders() {
    fetch('api.php?action=getProviders')
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                providers = data.providers;
                renderProviderList();
            } else {
                showToast('加载服务商列表失败', 'error');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showToast('网络错误', 'error');
        });
}

// 渲染服务商列表
function renderProviderList() {
    const listContainer = document.getElementById('providerList');
    
    if (providers.length === 0) {
        listContainer.innerHTML = `
            <div class="text-center py-4">
                <i class="bi bi-building text-muted" style="font-size: 3rem;"></i>
                <p class="text-muted mt-2">暂无服务商，点击上方按钮添加第一个服务商</p>
            </div>
        `;
        return;
    }
    
    let html = `
        <div class="table-responsive">
            <table class="table table-hover">
                <thead>
                    <tr>
                        <th>服务商名称</th>
                        <th>创建时间</th>
                        <th>操作</th>
                    </tr>
                </thead>
                <tbody>
    `;
    
    providers.forEach(provider => {
        html += `
            <tr>
                <td><strong>${escapeHtml(provider.name)}</strong></td>
                <td>${formatDateTime(provider.created_at)}</td>
                <td>
                    <button class="btn btn-sm btn-outline-primary me-1" onclick="editProvider(${provider.id})">
                        <i class="bi bi-pencil"></i> 编辑
                    </button>
                    <button class="btn btn-sm btn-outline-danger" onclick="deleteProvider(${provider.id}, '${escapeHtml(provider.name)}')">
                        <i class="bi bi-trash"></i> 删除
                    </button>
                </td>
            </tr>
        `;
    });
    
    html += `
                </tbody>
            </table>
        </div>
    `;
    
    listContainer.innerHTML = html;
}

// 显示添加服务商模态框
function showAddProviderModal() {
    document.getElementById('providerModalTitle').textContent = '添加服务商';
    document.getElementById('providerForm').reset();
    document.getElementById('providerId').value = '';
    
    const modal = new bootstrap.Modal(document.getElementById('providerModal'));
    modal.show();
}

// 编辑服务商
function editProvider(id) {
    const provider = providers.find(p => p.id === id);
    if (!provider) return;
    
    document.getElementById('providerModalTitle').textContent = '编辑服务商';
    document.getElementById('providerId').value = provider.id;
    document.getElementById('providerName').value = provider.name;
    
    const modal = new bootstrap.Modal(document.getElementById('providerModal'));
    modal.show();
}

// 保存服务商
function saveProvider() {
    const formData = new FormData(document.getElementById('providerForm'));
    const providerId = formData.get('providerId');
    
    const data = {
        action: providerId ? 'updateProvider' : 'addProvider',
        id: providerId || undefined,
        name: formData.get('providerName')
    };
    
    console.log('Saving provider:', data);
    
    fetch('api.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify(data)
    })
    .then(response => {
        console.log('Response status:', response.status);
        return response.json();
    })
    .then(data => {
        console.log('Response data:', data);
        if (data.success) {
            showToast(providerId ? '服务商更新成功' : '服务商添加成功', 'success');
            bootstrap.Modal.getInstance(document.getElementById('providerModal')).hide();
            loadProviders();
        } else {
            showToast(data.message || '操作失败', 'error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showToast('网络错误: ' + error.message, 'error');
    });
}

// 删除服务商
function deleteProvider(id, name) {
    if (!confirm(`确定要删除服务商 "${name}" 吗？\n\n注意：删除后，使用该服务商的服务器将显示为"未知服务商"。`)) {
        return;
    }
    
    // 防止重复点击
    const deleteButton = event.target.closest('button');
    if (deleteButton) {
        deleteButton.disabled = true;
    }
    
    fetch('api.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({
            action: 'deleteProvider',
            id: id
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showToast('服务商删除成功', 'success');
            loadProviders();
        } else {
            showToast(data.message || '删除失败', 'error');
            // 重新启用按钮
            if (deleteButton) {
                deleteButton.disabled = false;
            }
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showToast('网络错误', 'error');
        // 重新启用按钮
        if (deleteButton) {
            deleteButton.disabled = false;
        }
    });
}

// 工具函数
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

function formatDateTime(dateString) {
    const date = new Date(dateString);
    return date.toLocaleString('zh-CN', {
        year: 'numeric',
        month: '2-digit',
        day: '2-digit',
        hour: '2-digit',
        minute: '2-digit'
    });
}

// Toast 提示函数
function showToast(message, type = 'info') {
    // 创建 toast 容器（如果不存在）
    let toastContainer = document.getElementById('toast-container');
    if (!toastContainer) {
        toastContainer = document.createElement('div');
        toastContainer.id = 'toast-container';
        toastContainer.className = 'toast-container position-fixed top-0 end-0 p-3';
        toastContainer.style.zIndex = '9999';
        document.body.appendChild(toastContainer);
    }
    
    // 创建 toast 元素
    const toastId = 'toast-' + Date.now();
    const iconClass = type === 'success' ? 'bi-check-circle-fill text-success' : 
                     type === 'error' ? 'bi-exclamation-triangle-fill text-danger' : 
                     'bi-info-circle-fill text-primary';
    
    const toastHtml = `
        <div id="${toastId}" class="toast" role="alert">
            <div class="toast-header">
                <i class="bi ${iconClass} me-2"></i>
                <strong class="me-auto">${type === 'success' ? '成功' : type === 'error' ? '错误' : '提示'}</strong>
                <button type="button" class="btn-close" data-bs-dismiss="toast"></button>
            </div>
            <div class="toast-body">
                ${message}
            </div>
        </div>
    `;
    
    toastContainer.insertAdjacentHTML('beforeend', toastHtml);
    
    // 显示 toast
    const toastElement = document.getElementById(toastId);
    const toast = new bootstrap.Toast(toastElement, {
        autohide: true,
        delay: 3000
    });
    
    toast.show();
    
    // 自动清理
    toastElement.addEventListener('hidden.bs.toast', function() {
        toastElement.remove();
    });
}