/**
 * 货币管理类
 */
class CurrencyManager {
    constructor() {
        this.currencies = {};
        this.currentEditCode = null;
        // 货币中文名称映射
        this.currencyNames = {
            'CNY': '人民币',
            'USD': '美元',
            'EUR': '欧元',
            'GBP': '英镑',
            'HKD': '港元',
            'THB': '泰铢'
        };
        this.init();
    }

    /**
     * 初始化
     */
    init() {
        this.loadCurrencies();
        this.bindEvents();
    }

    /**
     * 绑定事件
     */
    bindEvents() {
        // 模态框关闭时重置表单
        document.getElementById('currencyModal').addEventListener('hidden.bs.modal', () => {
            this.resetForm();
        });

        // 表单提交事件
        document.getElementById('currencyForm').addEventListener('submit', (e) => {
            e.preventDefault();
            this.saveCurrency();
        });
    }

    /**
     * 加载货币列表
     */
    loadCurrencies() {
        this.showLoading();
        fetch('api.php?action=rates')
            .then(response => {
                console.log('Response status:', response.status);
                if (response.status === 401) {
                    // 未授权，重定向到登录页面
                    this.hideLoading();
                    window.location.href = 'login.php';
                    return;
                }
                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }
                return response.json();
            })
            .then(data => {
                if (!data) return; // 如果重定向了，data可能为undefined
                console.log('API response:', data);
                this.hideLoading();
                if (data.success) {
                    this.currencies = data.data;
                    this.currencyNames = data.names || {};
                    this.renderCurrencyList();
                } else {
                    this.showAlert('danger', data.message || '加载货币列表失败');
                }
            })
            .catch(error => {
                console.error('Fetch error:', error);
                this.hideLoading();
                this.showAlert('danger', '加载货币列表失败: ' + error.message);
            });
    }

    /**
     * 渲染货币列表
     */
    renderCurrencyList() {
        const currencyList = document.getElementById('currencyList');
        currencyList.innerHTML = '';

        // 获取货币代码并排序，CNY排在第一位
        const currencyCodes = Object.keys(this.currencies).sort((a, b) => {
            if (a === 'CNY') return -1;
            if (b === 'CNY') return 1;
            return a.localeCompare(b);
        });

        if (currencyCodes.length === 0) {
            currencyList.innerHTML = '<tr><td colspan="4" class="text-center">暂无货币数据</td></tr>';
            return;
        }

        // 渲染所有货币
        for (const code of currencyCodes) {
            const rate = this.currencies[code];
            const chineseName = this.currencyNames[code] || '未知货币';
            const updatedAt = new Date().toLocaleString();
            const isCNY = code === 'CNY';

            currencyList.innerHTML += `
                <tr>
                    <td>
                        <strong>${code}</strong>
                        ${this.currencyNames[code] ? `<br><small class="text-muted">${this.currencyNames[code]}</small>` : ''}
                    </td>
                    <td><strong>${isCNY ? '1.0000 (基准)' : parseFloat(rate).toFixed(4)}</strong></td>
                    <td class="text-muted small">${isCNY ? '基准货币' : `1${code} = ${parseFloat(rate).toFixed(4)}人民币`}</td>
                    <td>
                        ${isCNY ? 
                            '<span class="text-muted">基准货币</span>' : 
                            `<button class="btn btn-sm btn-outline-primary me-2" onclick="currencyManager.editCurrency('${code}')"><i class="fas fa-edit"></i> 编辑</button>
                             <button class="btn btn-sm btn-outline-danger" onclick="currencyManager.deleteCurrency('${code}')"><i class="fas fa-trash"></i> 删除</button>`
                        }
                    </td>
                </tr>
            `;
        }
    }

    /**
     * 显示添加货币模态框
     */
    showAddModal() {
        document.getElementById('currencyModalTitle').textContent = '添加货币';
        document.getElementById('editMode').value = 'add';
        document.getElementById('currencyCode').readOnly = false;
        document.getElementById('currencyCode').value = '';
        document.getElementById('currencyName').value = '';
        
        // 显示模态框
        const modalElement = document.getElementById('currencyModal');
        let modal = bootstrap.Modal.getInstance(modalElement);
        if (!modal) {
            modal = new bootstrap.Modal(modalElement);
        }
        modal.show();
    }

    /**
     * 编辑货币
     * @param {string} code 货币代码
     */
    editCurrency(code) {
        if (!this.currencies[code]) return;

        this.currentEditCode = code;
        document.getElementById('currencyModalTitle').textContent = '编辑货币';
        document.getElementById('editMode').value = 'edit';
        document.getElementById('currencyCode').value = code;
        document.getElementById('currencyCode').readOnly = true;
        document.getElementById('currencyName').value = this.currencyNames[code] || '';

        // 显示模态框
        const modalElement = document.getElementById('currencyModal');
        let modal = bootstrap.Modal.getInstance(modalElement);
        if (!modal) {
            modal = new bootstrap.Modal(modalElement);
        }
        modal.show();
    }

    /**
     * 保存货币
     */
    saveCurrency() {
        const form = document.getElementById('currencyForm');
        if (!form.checkValidity()) {
            form.reportValidity();
            return;
        }

        const editMode = document.getElementById('editMode').value;
        const code = document.getElementById('currencyCode').value.toUpperCase();
        const name = document.getElementById('currencyName').value;

        if (code === 'CNY') {
            this.showAlert('warning', 'CNY是基准货币，不能修改');
            return;
        }

        if (editMode === 'add' && this.currencies[code]) {
            this.showAlert('warning', `货币 ${code} 已存在`);
            return;
        }

        this.showLoading();

        // 准备请求数据
        const formData = new FormData();
        formData.append('action', 'saveCurrency');
        formData.append('code', code);
        formData.append('name', name);
        // 不再发送汇率，由后端自动获取

        fetch('api.php', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            this.hideLoading();
            if (data.success) {
                // 更新本地数据
                this.currencies[code] = data.rate || 0;
                this.currencyNames[code] = name;
                this.renderCurrencyList();
                
                // 关闭模态框
                const modal = bootstrap.Modal.getInstance(document.getElementById('currencyModal'));
                modal.hide();
                
                // 显示成功消息
                this.showAlert('success', editMode === 'add' ? '💰 货币添加成功！汇率已自动获取' : '✅ 货币更新成功！汇率信息已保存');
            } else {
                this.showAlert('danger', data.message || '❌ 保存货币失败，请检查输入信息后重试');
            }
        })
        .catch(error => {
            this.hideLoading();
            this.showAlert('danger', '💥 网络错误，保存失败: ' + error.message);
        });
    }

    /**
     * 删除货币
     * @param {string} code 货币代码
     */
    deleteCurrency(code) {
        if (code === 'CNY') {
            this.showAlert('warning', '🚫 CNY是基准货币，不能删除');
            return;
        }

        if (!confirm(`确定要删除货币 ${code} 吗？`)) {
            return;
        }

        // 防止重复点击
        const deleteButton = event.target.closest('button');
        if (deleteButton) {
            deleteButton.disabled = true;
        }

        this.showLoading();

        const formData = new FormData();
        formData.append('action', 'deleteCurrency');
        formData.append('code', code);

        fetch('api.php', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            this.hideLoading();
            if (data.success) {
                // 从本地数据中删除
                delete this.currencies[code];
                this.renderCurrencyList();
                this.showAlert('success', `🗑️ 货币 ${code} 删除成功！已从系统中移除`);
            } else {
                this.showAlert('danger', data.message || '❌ 删除货币失败，请稍后重试');
                // 重新启用按钮
                if (deleteButton) {
                    deleteButton.disabled = false;
                }
            }
        })
        .catch(error => {
            this.hideLoading();
            this.showAlert('danger', '💥 网络错误，删除失败: ' + error.message);
            // 重新启用按钮
            if (deleteButton) {
                deleteButton.disabled = false;
            }
        });
    }

    /**
     * 重置表单
     */
    resetForm() {
        document.getElementById('currencyForm').reset();
        this.currentEditCode = null;
    }

    /**
     * 显示提示信息
     * @param {string} type 提示类型 (success, danger, warning, info)
     * @param {string} message 提示消息
     */
    showAlert(type, message) {
        const alertToast = document.getElementById('alertToast');
        const alertTitle = document.getElementById('alertTitle');
        const alertMessage = document.getElementById('alertMessage');

        // 设置提示类型和消息
        alertToast.className = `toast border-${type}`;
        alertTitle.className = `me-auto text-${type}`;
        alertTitle.textContent = type === 'success' ? '成功' : type === 'danger' ? '错误' : '提示';
        alertMessage.textContent = message;

        // 显示提示
        const toast = new bootstrap.Toast(alertToast);
        toast.show();
    }

    /**
     * 显示加载指示器
     */
    showLoading() {
        document.getElementById('loadingIndicator').style.display = 'flex';
    }

    /**
     * 隐藏加载指示器
     */
    hideLoading() {
        document.getElementById('loadingIndicator').style.display = 'none';
    }

    /**
     * 一键更新所有汇率
     */
    updateAllRates() {
        const updateBtn = document.getElementById('updateRatesBtn');
        const originalText = updateBtn.innerHTML;
        
        // 禁用按钮并显示加载状态
        updateBtn.disabled = true;
        updateBtn.innerHTML = '<i class="bi bi-arrow-clockwise spin"></i> 更新中...';
        
        fetch('api.php?action=updateAllRates', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                this.showAlert('success', '🔄 汇率更新成功！所有货币汇率已同步到最新数据');
                this.loadCurrencies(); // 重新加载货币列表
            } else {
                this.showAlert('danger', '❌ 汇率更新失败: ' + (data.message || '未知错误'));
            }
        })
        .catch(error => {
            this.showAlert('danger', '💥 网络错误，汇率更新失败: ' + error.message);
        })
        .finally(() => {
            // 恢复按钮状态
            updateBtn.disabled = false;
            updateBtn.innerHTML = originalText;
        });
    }
}

// 初始化货币管理器
let currencyManager;
document.addEventListener('DOMContentLoaded', function() {
    currencyManager = new CurrencyManager();
});