<?php
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// 检查登录状态
if (!isset($_SESSION['logged_in']) || $_SESSION['logged_in'] !== true) {
    http_response_code(401);
    echo json_encode(['error' => '未授权访问']);
    exit();
}

header('Content-Type: application/json');

// 获取action参数
$input = json_decode(file_get_contents('php://input'), true);
$action = $_GET['action'] ?? $_POST['action'] ?? $input['action'] ?? '';
$dataDir = 'data';
$serversFile = $dataDir . '/servers.json';
$providersFile = $dataDir . '/providers.json';
$categoriesFile = $dataDir . '/categories.json';

// 确保数据目录存在
if (!is_dir($dataDir)) {
    mkdir($dataDir, 0755, true);
}

// 读取服务器数据
function getServers() {
    global $serversFile;
    if (!file_exists($serversFile)) {
        return [];
    }
    $data = file_get_contents($serversFile);
    return json_decode($data, true) ?: [];
}

// 保存服务器数据
function saveServers($servers) {
    global $serversFile;
    return file_put_contents($serversFile, json_encode($servers, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

// 保存单个服务器数据
function saveServer($data) {
    $servers = getServers();
    $id = $data['id'] ?? generateId();
    // 处理分类数据
    $categories = [];
    if (isset($data['categories'])) {
        if (is_string($data['categories'])) {
            $categories = json_decode($data['categories'], true) ?: [];
        } else if (is_array($data['categories'])) {
            $categories = $data['categories'];
        }
    } else if (isset($data['category'])) {
        $categories = [$data['category']];
    }
    
    if (empty($categories)) {
        $categories = ['其它'];
    }
    
    $serverData = [
        'id' => $id,
        'serverName' => $data['serverName'],
        'provider' => $data['provider'],
        'categories' => $categories,
        'category' => $categories[0], // 保持向后兼容
        'cost' => floatval($data['cost']),
        'currency' => $data['currency'],
        'purchaseDate' => $data['purchaseDate'],
        'expireDate' => $data['expireDate'],
        'renewStatus' => $data['renewStatus'] ?? 'active',
        'autoRenew' => isset($data['autoRenew']) ? (bool)$data['autoRenew'] : false,
        'renewCycle' => intval($data['renewCycle'] ?? 30),
        'panelAccount' => $data['panelAccount'] ?? '',
        'panelPassword' => $data['panelPassword'] ?? '',
        'notes' => $data['notes'] ?? '',
        'createdAt' => $data['createdAt'] ?? date('Y-m-d H:i:s'),
        'updatedAt' => date('Y-m-d H:i:s')
    ];
    
    // 查找并更新现有服务器或添加新服务器
    $found = false;
    foreach ($servers as &$server) {
        if ($server['id'] === $id) {
            $server = $serverData;
            $found = true;
            break;
        }
    }
    
    if (!$found) {
        $servers[] = $serverData;
    }
    
    saveServers($servers);
    return $serverData;
}

// 获取单个服务器数据
function getServer($id) {
    $servers = getServers();
    foreach ($servers as $server) {
        if ($server['id'] === $id) {
            return $server;
        }
    }
    return null;
}

// 生成唯一ID
function generateId() {
    return uniqid('server_', true);
}

// 服务商管理函数
function getProviders() {
    global $providersFile;
    if (!file_exists($providersFile)) {
        return [];
    }
    $data = file_get_contents($providersFile);
    return json_decode($data, true) ?: [];
}

function saveProviders($providers) {
    global $providersFile;
    return file_put_contents($providersFile, json_encode($providers, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

function saveProvider($data) {
    $providers = getProviders();
    $id = $data['id'] ?? time();
    $providerData = [
        'id' => $id,
        'name' => $data['name'],
        'created_at' => $data['created_at'] ?? date('Y-m-d H:i:s'),
        'updated_at' => date('Y-m-d H:i:s')
    ];
    
    $found = false;
    foreach ($providers as &$provider) {
        if ($provider['id'] === $id) {
            $provider = $providerData;
            $found = true;
            break;
        }
    }
    
    if (!$found) {
        $providers[] = $providerData;
    }
    
    saveProviders($providers);
    return $providerData;
}

// 分类管理函数
function getCategories() {
    global $categoriesFile;
    if (!file_exists($categoriesFile)) {
        return [];
    }
    $data = file_get_contents($categoriesFile);
    $categories = json_decode($data, true) ?: [];
    
    // 为每个分类添加服务器数量
    $servers = getServers();
    foreach ($categories as &$category) {
        $count = 0;
        foreach ($servers as $server) {
            if ($server['category'] === $category['name']) {
                $count++;
            }
        }
        $category['server_count'] = $count;
    }
    
    return $categories;
}

function saveCategories($categories) {
    global $categoriesFile;
    return file_put_contents($categoriesFile, json_encode($categories, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

function saveCategory($data) {
    $categories = getCategories();
    $id = $data['id'] ?? null;
    
    if ($id) {
        // 更新现有分类
        $found = false;
        foreach ($categories as &$category) {
            if ($category['id'] == $id) {
                $category['name'] = $data['name'];
                $category['color'] = $data['color'] ?? '#0d6efd';
                $category['description'] = $data['description'] ?? '';
                $category['updated_at'] = date('Y-m-d H:i:s');
                $found = true;
                $categoryData = $category;
                break;
            }
        }
        if (!$found) {
            return false; // 分类不存在
        }
    } else {
        // 添加新分类
        $id = time();
        $categoryData = [
            'id' => $id,
            'name' => $data['name'],
            'color' => $data['color'] ?? '#0d6efd',
            'description' => $data['description'] ?? '',
            'created_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s')
        ];
        $categories[] = $categoryData;
    }
    
    saveCategories($categories);
    return $categoryData;
}

// 获取汇率（1外币兑换人民币）
function getExchangeRates() {
    $ratesFile = 'data/exchange_rates.json';
    
    // 直接读取本地汇率文件
    if (file_exists($ratesFile)) {
        $rates = json_decode(file_get_contents($ratesFile), true);
        return $rates ?: [];
    }
    
    // 如果文件不存在，返回默认汇率（1外币兑换人民币）
    return [
        'CNY' => 1,
        'USD' => 7.2,
        'EUR' => 7.8,
        'GBP' => 9.1,
        'HKD' => 0.92,
        'THB' => 0.20
    ];
}

// 保存货币汇率和名称
function saveCurrency($code, $rate, $name = '') {
    if ($code === 'CNY') {
        return ['success' => false, 'message' => 'CNY是基准货币，不能修改'];
    }
    
    // 保存汇率
    $rates = getExchangeRates();
    $rates[$code] = (float)$rate;
    
    $ratesFile = 'data/exchange_rates.json';
    $ratesSaved = file_put_contents($ratesFile, json_encode($rates, JSON_PRETTY_PRINT));
    
    // 保存货币名称
    $namesSaved = true;
    if (!empty($name)) {
        $namesFile = 'data/currency_names.json';
        $names = [];
        if (file_exists($namesFile)) {
            $names = json_decode(file_get_contents($namesFile), true) ?: [];
        }
        $names[$code] = $name;
        $namesSaved = file_put_contents($namesFile, json_encode($names, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    }
    
    if ($ratesSaved && $namesSaved) {
        return ['success' => true, 'message' => '货币保存成功', 'rate' => $rate];
    }
    
    return ['success' => false, 'message' => '保存货币失败'];
}

// 删除货币
function deleteCurrency($code) {
    if ($code === 'CNY') {
        return ['success' => false, 'message' => 'CNY是基准货币，不能删除'];
    }
    
    $rates = getExchangeRates();
    if (!isset($rates[$code])) {
        return ['success' => false, 'message' => '货币不存在'];
    }
    
    unset($rates[$code]);
    
    $ratesFile = 'data/exchange_rates.json';
    if (file_put_contents($ratesFile, json_encode($rates, JSON_PRETTY_PRINT))) {
        return ['success' => true, 'message' => '货币删除成功'];
    }
    
    return ['success' => false, 'message' => '删除货币失败'];
}

// 续期服务器
function renewServer($id, $newExpireDate) {
    $servers = getServers();
    $updated = false;
    
    foreach ($servers as &$server) {
        if ($server['id'] === $id) {
            // 如果已归档，不允许续期
            if ($server['renewStatus'] === 'archived') {
                return ['success' => false, 'message' => '已归档的服务器不能续期'];
            }
            
            $server['expireDate'] = $newExpireDate;
            $server['renewStatus'] = 'active';
            $server['updatedAt'] = date('Y-m-d H:i:s');
            $updated = true;
            break;
        }
    }
    
    if ($updated) {
        saveServers($servers);
        return ['success' => true, 'message' => '服务器已成功续期'];
    }
    
    return ['success' => false, 'message' => '找不到指定的服务器'];
}

switch ($action) {
    case 'list':
        $servers = getServers();
        $rates = getExchangeRates();
        
        // 计算每个服务器的CNY费用
        foreach ($servers as &$server) {
            $currency = $server['currency'];
            $cost = floatval($server['cost']);
            
            if ($currency === 'CNY') {
                $server['cost_cny'] = $cost;
            } else {
                $rate = $rates[$currency] ?? 1;
                $server['cost_cny'] = $cost * $rate;
            }
            
            // 计算剩余天数
            $expireDate = new DateTime($server['expireDate']);
            $today = new DateTime();
            $interval = $today->diff($expireDate);
            $server['days_remaining'] = $expireDate > $today ? $interval->days : -$interval->days;
        }
        
        echo json_encode(['success' => true, 'data' => $servers, 'rates' => $rates]);
        break;
        
    case 'add':
        // 处理分类数据
        $categories = [];
        if (isset($_POST['categories'])) {
            if (is_string($_POST['categories'])) {
                $categories = json_decode($_POST['categories'], true) ?: [];
            } else if (is_array($_POST['categories'])) {
                $categories = $_POST['categories'];
            }
        } else if (isset($_POST['category'])) {
            $categories = [$_POST['category']];
        }
        
        if (empty($categories)) {
            $categories = ['其它'];
        }
        
        $data = [
            'serverName' => $_POST['serverName'] ?? '',
            'provider' => $_POST['provider'] ?? '',
            'categories' => $categories,
            'category' => $categories[0], // 保持向后兼容
            'cost' => $_POST['cost'] ?? 0,
            'currency' => $_POST['currency'] ?? 'USD',
            'purchaseDate' => $_POST['purchaseDate'] ?? '',
            'expireDate' => $_POST['expireDate'] ?? '',
            'renewStatus' => $_POST['renewStatus'] ?? 'active',
            'autoRenew' => isset($_POST['autoRenew']) ? true : false,
            'renewCycle' => $_POST['renewCycle'] ?? 30,
            'notes' => $_POST['notes'] ?? ''
        ];
        $result = saveServer($data);
        echo json_encode(['success' => true, 'data' => $result, 'message' => '服务器添加成功']);
        break;
        
    case 'update':
        $serverId = $_POST['serverId'] ?? '';
        
        // 检查服务器是否已归档
        $existingData = getServer($serverId);
        if ($existingData && $existingData['renewStatus'] === 'archived') {
            echo json_encode(['success' => false, 'message' => '已归档的服务器不允许修改']);
            break;
        }
        
        // 处理分类数据
        $categories = [];
        if (isset($_POST['categories'])) {
            if (is_string($_POST['categories'])) {
                $categories = json_decode($_POST['categories'], true) ?: [];
            } else if (is_array($_POST['categories'])) {
                $categories = $_POST['categories'];
            }
        } else if (isset($_POST['category'])) {
            $categories = [$_POST['category']];
        }
        
        if (empty($categories)) {
            $categories = ['其它'];
        }
        
        $data = [
            'id' => $serverId,
            'serverName' => $_POST['serverName'] ?? '',
            'provider' => $_POST['provider'] ?? '',
            'categories' => $categories,
            'category' => $categories[0], // 保持向后兼容
            'cost' => $_POST['cost'] ?? 0,
            'currency' => $_POST['currency'] ?? 'USD',
            'purchaseDate' => $_POST['purchaseDate'] ?? '',
            'expireDate' => $_POST['expireDate'] ?? '',
            'renewStatus' => $_POST['renewStatus'] ?? 'active',
            'autoRenew' => isset($_POST['autoRenew']) ? true : false,
            'renewCycle' => $_POST['renewCycle'] ?? 30,
            'notes' => $_POST['notes'] ?? ''
        ];
        
        // 保留原始创建时间
        if ($existingData) {
            $data['createdAt'] = $existingData['createdAt'];
        }
        
        $result = saveServer($data);
        echo json_encode(['success' => true, 'data' => $result, 'message' => '服务器更新成功']);
        break;
        
    case 'delete':
        $serverId = $_POST['serverId'] ?? $_GET['serverId'] ?? '';
        
        // 检查服务器是否已归档
        $existingData = getServer($serverId);
        if ($existingData && $existingData['renewStatus'] === 'archived') {
            echo json_encode(['success' => false, 'message' => '已归档的服务器不允许删除']);
            break;
        }
        
        $servers = getServers();
        $servers = array_filter($servers, function($server) use ($serverId) {
            return $server['id'] !== $serverId;
        });
        
        if (saveServers(array_values($servers))) {
            echo json_encode(['success' => true, 'message' => '服务器删除成功']);
        } else {
            echo json_encode(['success' => false, 'message' => '删除失败']);
        }
        break;
        
    case 'rates':
        $rates = getExchangeRates();
        
        // 获取货币名称
        $namesFile = 'data/currency_names.json';
        $names = [];
        if (file_exists($namesFile)) {
            $names = json_decode(file_get_contents($namesFile), true) ?: [];
        }
        
        echo json_encode(['success' => true, 'data' => $rates, 'names' => $names]);
        break;
        
    case 'saveCurrency':
        $input = json_decode(file_get_contents('php://input'), true);
        $code = $_POST['code'] ?? $input['code'] ?? '';
        $name = $_POST['name'] ?? $input['name'] ?? '';
        
        if (empty($code)) {
            echo json_encode(['success' => false, 'message' => '货币代码不能为空']);
            break;
        }
        
        // 自动获取汇率（这里使用预设汇率，实际项目中可以调用汇率API）
        $defaultRates = [
            'USD' => 7.2,
            'EUR' => 7.8,
            'GBP' => 9.1,
            'HKD' => 0.92,
            'THB' => 0.20,
            'JPY' => 0.048,
            'KRW' => 0.0054,
            'SGD' => 5.3,
            'AUD' => 4.7,
            'CAD' => 5.2
        ];
        
        $rate = $defaultRates[$code] ?? 1;
        $result = saveCurrency($code, $rate, $name);
        echo json_encode($result);
        break;
        
    case 'updateAllRates':
        // 模拟从外部API获取最新汇率（这里使用固定汇率作为示例）
        $newRates = [
            'CNY' => 1,
            'USD' => 0.14,
            'EUR' => 0.12,
            'GBP' => 0.10,
            'HKD' => 1.09,
            'THB' => 4.51
        ];
        
        // 保存新汇率到文件
        $exchangeRatesFile = 'data/exchange_rates.json';
        $success = file_put_contents($exchangeRatesFile, json_encode($newRates, JSON_PRETTY_PRINT));
        
        if ($success !== false) {
            echo json_encode(['success' => true, 'message' => '汇率更新成功', 'data' => $newRates]);
        } else {
            echo json_encode(['success' => false, 'message' => '汇率更新失败']);
        }
        break;
        
    case 'deleteCurrency':
        $code = $_POST['code'] ?? '';
        
        if (empty($code)) {
            echo json_encode(['success' => false, 'message' => '货币代码不能为空']);
            break;
        }
        
        $result = deleteCurrency($code);
        echo json_encode($result);
        break;
        
    case 'archive':
        $serverId = $_POST['serverId'] ?? '';
        $existingData = getServer($serverId);
        
        if (!$existingData) {
            echo json_encode(['success' => false, 'message' => '服务器不存在']);
            break;
        }
        
        $existingData['renewStatus'] = 'archived';
        $result = saveServer($existingData);
        echo json_encode(['success' => true, 'data' => $result, 'message' => '服务器已归档']);
        break;
        
    case 'renew':
        $serverId = $_POST['serverId'] ?? '';
        if (empty($serverId)) {
            echo json_encode(['success' => false, 'message' => '服务器ID不能为空']);
            break;
        }
        
        // 获取服务器信息
        $servers = getServers();
        $server = null;
        foreach ($servers as $s) {
            if ($s['id'] === $serverId) {
                $server = $s;
                break;
            }
        }
        
        if (!$server) {
            echo json_encode(['success' => false, 'message' => '服务器不存在']);
            break;
        }
        
        // 计算新的到期日期（当前到期日期 + 30天）
        $expireDate = new DateTime($server['expireDate']);
        $expireDate->modify('+30 days');
        $newExpireDate = $expireDate->format('Y-m-d');
        
        // 调用续期函数
        $result = renewServer($serverId, $newExpireDate);
        echo json_encode($result);
        break;
        
    case 'auto_renew_check':
        // 检查并处理自动续期
        $servers = getServers();
        $renewedCount = 0;
        $today = new DateTime();
        
        foreach ($servers as &$server) {
            // 只处理开启了自动续期且状态为活跃的服务器
            if (!isset($server['autoRenew']) || !$server['autoRenew'] || $server['renewStatus'] !== 'active') {
                continue;
            }
            
            $expireDate = new DateTime($server['expireDate']);
            
            // 如果已经到期或今天到期，则进行自动续期
            if ($expireDate <= $today) {
                $renewCycle = intval($server['renewCycle'] ?? 30);
                $newExpireDate = clone $expireDate;
                $newExpireDate->modify("+{$renewCycle} days");
                
                $server['expireDate'] = $newExpireDate->format('Y-m-d');
                $server['updatedAt'] = date('Y-m-d H:i:s');
                $renewedCount++;
            }
        }
        
        if ($renewedCount > 0) {
            saveServers($servers);
        }
        
        echo json_encode([
            'success' => true, 
            'message' => "自动续期检查完成，共续期 {$renewedCount} 台服务器",
            'renewed_count' => $renewedCount
        ]);
        break;
        
    // 服务商管理接口
    case 'getProviders':
        $providers = getProviders();
        echo json_encode(['success' => true, 'providers' => $providers]);
        break;
        
    case 'addProvider':
        $data = [
            'name' => $input['name'] ?? ''
        ];
        
        if (empty($data['name'])) {
            echo json_encode(['success' => false, 'message' => '服务商名称不能为空']);
            break;
        }
        
        $result = saveProvider($data);
        echo json_encode(['success' => true, 'data' => $result, 'message' => '服务商添加成功']);
        break;
        
    case 'updateProvider':
        $input = json_decode(file_get_contents('php://input'), true);
        $data = [
            'id' => $input['id'] ?? '',
            'name' => $input['name'] ?? ''
        ];
        
        if (empty($data['id']) || empty($data['name'])) {
            echo json_encode(['success' => false, 'message' => '服务商ID和名称不能为空']);
            break;
        }
        
        $result = saveProvider($data);
        echo json_encode(['success' => true, 'data' => $result, 'message' => '服务商更新成功']);
        break;
        
    case 'deleteProvider':
        $input = json_decode(file_get_contents('php://input'), true);
        $id = $input['id'] ?? '';
        
        if (empty($id)) {
            echo json_encode(['success' => false, 'message' => '服务商ID不能为空']);
            break;
        }
        
        $providers = getProviders();
        $providers = array_filter($providers, function($provider) use ($id) {
            return $provider['id'] != $id;
        });
        
        if (saveProviders(array_values($providers))) {
            echo json_encode(['success' => true, 'message' => '服务商删除成功']);
        } else {
            echo json_encode(['success' => false, 'message' => '删除失败']);
        }
        break;
        
    // 服务商管理接口
    case 'getProviders':
        $providers = getProviders();
        echo json_encode(['success' => true, 'providers' => $providers]);
        break;
        
    // 分类管理接口
    case 'getCategories':
        $categories = getCategories();
        echo json_encode(['success' => true, 'categories' => $categories]);
        break;
        
    case 'addCategory':
        $input = json_decode(file_get_contents('php://input'), true);
        $data = [
            'name' => $input['name'] ?? '',
            'color' => $input['color'] ?? '#0d6efd',
            'description' => $input['description'] ?? ''
        ];
        
        if (empty($data['name'])) {
            echo json_encode(['success' => false, 'message' => '分类名称不能为空']);
            break;
        }
        
        $result = saveCategory($data);
        echo json_encode(['success' => true, 'data' => $result, 'message' => '分类添加成功']);
        break;
        
    case 'updateCategory':
        $input = json_decode(file_get_contents('php://input'), true);
        $data = [
            'id' => $input['id'] ?? '',
            'name' => $input['name'] ?? '',
            'color' => $input['color'] ?? '#0d6efd',
            'description' => $input['description'] ?? ''
        ];
        
        if (empty($data['id']) || empty($data['name'])) {
            echo json_encode(['success' => false, 'message' => '分类ID和名称不能为空']);
            break;
        }
        
        $result = saveCategory($data);
        echo json_encode(['success' => true, 'data' => $result, 'message' => '分类更新成功']);
        break;
        
    case 'deleteCategory':
        $input = json_decode(file_get_contents('php://input'), true);
        $id = $input['id'] ?? '';
        
        if (empty($id)) {
            echo json_encode(['success' => false, 'message' => '分类ID不能为空']);
            break;
        }
        
        $categories = getCategories();
        $categories = array_filter($categories, function($category) use ($id) {
            return $category['id'] != $id;
        });
        
        if (saveCategories(array_values($categories))) {
            echo json_encode(['success' => true, 'message' => '分类删除成功']);
        } else {
            echo json_encode(['success' => false, 'message' => '删除失败']);
        }
        break;
        
    default:
        echo json_encode(['success' => false, 'message' => '无效的操作']);
        break;
}
?>