<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>商户管理 - 美团子账号管理系统</title>
    <link rel="icon" type="image/svg+xml" href="assets/favicon.svg">
    <link rel="stylesheet" href="assets/style.css">
    <style>
        .merchant-container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
        }
        
        .page-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
            padding-bottom: 20px;
            border-bottom: 2px solid #e9ecef;
        }
        
        .page-title {
            font-size: 24px;
            font-weight: 600;
            color: #333;
            margin: 0;
        }
        
        .nav-menu {
            display: flex;
            gap: 10px;
            align-items: center;
        }
        
        .nav-btn {
            background: #007bff;
            color: white;
            border: none;
            padding: 8px 16px;
            border-radius: 4px;
            cursor: pointer;
            font-size: 14px;
            text-decoration: none;
            transition: background-color 0.2s;
        }
        
        .nav-btn:hover {
            background: #0056b3;
        }
        
        .nav-btn.active {
            background: #28a745;
        }
        
        .back-btn {
            background: #6c757d;
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 4px;
            cursor: pointer;
            font-size: 14px;
            text-decoration: none;
            transition: background-color 0.2s;
        }
        
        .back-btn:hover {
            background: #5a6268;
        }
        
        .merchant-section {
            background: white;
            border-radius: 8px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
            overflow: hidden;
        }
        
        .section-header {
            background: #f8f9fa;
            padding: 20px;
            border-bottom: 1px solid #e9ecef;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .section-controls {
            display: flex;
            align-items: center;
            gap: 20px;
        }
        
        .filter-group {
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .filter-group label {
            font-size: 14px;
            color: #666;
            white-space: nowrap;
        }
        
        .filter-group select {
            padding: 6px 12px;
            border: 1px solid #ddd;
            border-radius: 4px;
            font-size: 14px;
            background: white;
            min-width: 120px;
        }
        
        .filter-group select:focus {
            outline: none;
            border-color: #007bff;
            box-shadow: 0 0 0 2px rgba(0,123,255,0.25);
        }
        
        .section-title {
            font-size: 18px;
            font-weight: 600;
            color: #333;
            margin: 0;
        }
        
        .add-btn {
            background: #007bff;
            color: white;
            border: none;
            padding: 8px 16px;
            border-radius: 4px;
            cursor: pointer;
            font-size: 14px;
            transition: background-color 0.2s;
        }
        
        .add-btn:hover {
            background: #0056b3;
        }
        
        .merchant-list {
            padding: 20px;
        }
        
        .floor-group {
            margin-bottom: 30px;
        }
        
        .floor-group:last-child {
            margin-bottom: 0;
        }
        
        .floor-title {
            font-size: 16px;
            font-weight: 600;
            color: #007bff;
            margin: 0 0 15px 0;
            padding: 10px 15px;
            background: #f8f9fa;
            border-left: 4px solid #007bff;
            border-radius: 4px;
        }
        
        .floor-merchants {
            padding-left: 15px;
        }
        
        .merchant-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 15px 0;
            border-bottom: 1px solid #f0f0f0;
        }
        
        .merchant-item:last-child {
            border-bottom: none;
        }
        
        .merchant-item-info {
            flex: 1;
        }
        
        .merchant-item-name {
            font-size: 16px;
            font-weight: 500;
            color: #333;
            margin-bottom: 5px;
        }
        
        .merchant-item-time {
            font-size: 12px;
            color: #666;
        }
        
        .merchant-item-detail {
            font-size: 12px;
            color: #888;
            margin-top: 2px;
        }
        
        .merchant-item-actions {
            display: flex;
            gap: 8px;
        }
        
        .edit-btn {
            background: #28a745;
            color: white;
            border: none;
            padding: 6px 12px;
            border-radius: 4px;
            cursor: pointer;
            font-size: 12px;
            transition: background-color 0.2s;
        }
        
        .edit-btn:hover {
            background: #218838;
        }
        
        .delete-btn {
            background: #dc3545;
            color: white;
            border: none;
            padding: 6px 12px;
            border-radius: 4px;
            cursor: pointer;
            font-size: 12px;
            transition: background-color 0.2s;
        }
        
        .delete-btn:hover {
            background: #c82333;
        }
        
        .empty-state {
            text-align: center;
            padding: 40px 20px;
            color: #666;
        }
        
        .empty-state img {
            width: 80px;
            height: 80px;
            margin-bottom: 16px;
            opacity: 0.5;
        }
        
        .message-modal, .confirm-modal {
            max-width: 400px;
            margin: 15% auto;
            animation: slideDown 0.3s ease-out;
        }
        
        .message-header, .confirm-header {
            display: flex;
            align-items: center;
            gap: 10px;
            padding: 20px 20px 10px;
            border-bottom: 1px solid #e9ecef;
        }
        
        .message-icon, .confirm-icon {
            font-size: 24px;
        }
        
        .message-title, .confirm-title {
            margin: 0;
            font-size: 18px;
            font-weight: 600;
        }
        
        .message-body, .confirm-body {
            padding: 20px;
        }
        
        .message-footer, .confirm-footer {
            padding: 10px 20px 20px;
            text-align: right;
            display: flex;
            gap: 10px;
            justify-content: flex-end;
        }
        
        @keyframes slideDown {
            from {
                opacity: 0;
                transform: translateY(-50px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        .btn-danger {
            background: #dc3545;
            color: white;
            border: none;
            padding: 8px 16px;
            border-radius: 4px;
            cursor: pointer;
            font-size: 14px;
            transition: background-color 0.2s;
        }
        
        .btn-danger:hover {
            background: #c82333;
        }
        
        /* 模态框样式 */
        .modal {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0,0,0,0.5);
            animation: fadeIn 0.3s ease;
        }
        
        .modal-content {
            background-color: white;
            margin: 10% auto;
            padding: 0;
            border-radius: 8px;
            width: 500px;
            max-width: 90%;
            box-shadow: 0 4px 20px rgba(0,0,0,0.15);
            animation: slideIn 0.3s ease;
        }
        
        .modal-header {
            padding: 20px;
            border-bottom: 1px solid #e9ecef;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .modal-title {
            font-size: 18px;
            font-weight: 600;
            color: #333;
            margin: 0;
        }
        
        .close {
            background: none;
            border: none;
            font-size: 24px;
            cursor: pointer;
            color: #666;
            padding: 0;
            width: 30px;
            height: 30px;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .close:hover {
            color: #333;
        }
        
        .modal-body {
            padding: 20px;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 8px;
            font-weight: 500;
            color: #333;
        }
        
        .form-group input,
        .form-group select {
            width: 100%;
            padding: 10px;
            border: 1px solid #ddd;
            border-radius: 4px;
            font-size: 14px;
            box-sizing: border-box;
        }
        
        .form-group input:focus,
        .form-group select:focus {
            outline: none;
            border-color: #007bff;
            box-shadow: 0 0 0 2px rgba(0,123,255,0.25);
        }
        
        .modal-footer {
            padding: 20px;
            border-top: 1px solid #e9ecef;
            display: flex;
            justify-content: flex-end;
            gap: 10px;
        }
        
        .btn {
            padding: 10px 20px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-size: 14px;
            transition: background-color 0.2s;
        }
        
        .btn-primary {
            background: #007bff;
            color: white;
        }
        
        .btn-primary:hover {
            background: #0056b3;
        }
        
        .btn-secondary {
            background: #6c757d;
            color: white;
        }
        
        .btn-secondary:hover {
            background: #5a6268;
        }
        
        @keyframes fadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }
        
        @keyframes slideIn {
            from { transform: translateY(-50px); opacity: 0; }
            to { transform: translateY(0); opacity: 1; }
        }
    </style>
</head>
<body>
    <div class="merchant-container">
        <div class="page-header">
            <h1 class="page-title">商户管理</h1>
            <div class="nav-menu">
                <a href="merchants.php" class="nav-btn active">商户管理</a>
                <a href="categories.php" class="nav-btn">分类管理</a>
                <a href="floors.php" class="nav-btn">楼层管理</a>
                <a href="index.php" class="back-btn">返回主页</a>
            </div>
        </div>
        
        <!-- 商户管理 -->
        <div class="merchant-section">
            <div class="section-header">
                <h2 class="section-title">商户列表</h2>
                <div class="section-controls">
                    <div class="filter-group">
                        <label for="floorFilter">按楼层筛选：</label>
                        <select id="floorFilter" onchange="filterByFloor()">
                            <option value="">全部楼层</option>
                        </select>
                    </div>
                    <button class="add-btn" onclick="openAddModal()">添加商户</button>
                </div>
            </div>
            <div class="merchant-list" id="merchantList">
                <div class="empty-state">
                    <img src="assets/empty-data.svg" alt="暂无数据">
                    <p>暂无商户数据</p>
                </div>
            </div>
        </div>
    </div>
    
    <!-- 添加/编辑模态框 -->
    <div id="merchantModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 class="modal-title" id="modalTitle">添加商户</h3>
                <button class="close" onclick="closeModal()">&times;</button>
            </div>
            <div class="modal-body">
                <form id="merchantForm">
                    <div class="form-group">
                        <label for="merchantName">商户名称</label>
                        <input type="text" id="merchantName" name="name" required>
                    </div>
                    <div class="form-group">
                        <label for="merchantCategory">分类</label>
                        <select id="merchantCategory" name="category" required>
                            <option value="">请选择分类</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="merchantFloor">楼层</label>
                        <select id="merchantFloor" name="floor" required>
                            <option value="">请选择楼层</option>
                        </select>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" onclick="closeModal()">取消</button>
                <button type="button" class="btn btn-primary" onclick="saveMerchant()">保存</button>
            </div>
        </div>
    </div>
    
    <!-- 消息提示弹窗 -->
    <div id="messageModal" class="modal" style="display: none;">
        <div class="modal-content message-modal">
            <div class="message-header">
                <span id="messageIcon" class="message-icon"></span>
                <h3 id="messageTitle" class="message-title"></h3>
            </div>
            <div class="message-body">
                <p id="messageText"></p>
            </div>
            <div class="message-footer">
                <button type="button" class="btn btn-primary" onclick="closeMessageModal()">确定</button>
            </div>
        </div>
    </div>

    <!-- 确认对话框 -->
    <div id="confirmModal" class="modal" style="display: none;">
        <div class="modal-content confirm-modal">
            <div class="confirm-header">
                <span class="confirm-icon">⚠️</span>
                <h3 class="confirm-title">确认操作</h3>
            </div>
            <div class="confirm-body">
                <p id="confirmText"></p>
            </div>
            <div class="confirm-footer">
                <button type="button" class="btn btn-secondary" onclick="closeConfirmModal()">取消</button>
                <button type="button" class="btn btn-danger" id="confirmBtn" onclick="confirmAction()">确定</button>
            </div>
        </div>
    </div>

    <script>
        let currentId = '';
        let isEditMode = false;
        let confirmCallback = null;
        let allMerchants = []; // 存储所有商户数据
        let allFloors = []; // 存储所有楼层数据
        
        // 显示消息提示
        function showMessage(message, type = 'info') {
            const modal = document.getElementById('messageModal');
            const icon = document.getElementById('messageIcon');
            const title = document.getElementById('messageTitle');
            const text = document.getElementById('messageText');
            
            // 设置图标和标题
            switch (type) {
                case 'success':
                    icon.textContent = '✅';
                    title.textContent = '操作成功';
                    break;
                case 'error':
                    icon.textContent = '❌';
                    title.textContent = '操作失败';
                    break;
                case 'warning':
                    icon.textContent = '⚠️';
                    title.textContent = '警告';
                    break;
                default:
                    icon.textContent = 'ℹ️';
                    title.textContent = '提示';
            }
            
            text.textContent = message;
            modal.style.display = 'block';
            
            // 3秒后自动关闭成功消息
            if (type === 'success') {
                setTimeout(() => {
                    closeMessageModal();
                }, 3000);
            }
        }
        
        // 关闭消息提示
        function closeMessageModal() {
            document.getElementById('messageModal').style.display = 'none';
        }
        
        // 显示确认对话框
        function showConfirm(message, callback) {
            document.getElementById('confirmText').textContent = message;
            document.getElementById('confirmModal').style.display = 'block';
            confirmCallback = callback;
        }
        
        // 关闭确认对话框
        function closeConfirmModal() {
            document.getElementById('confirmModal').style.display = 'none';
            confirmCallback = null;
        }
        
        // 确认操作
        function confirmAction() {
            if (confirmCallback) {
                confirmCallback();
            }
            closeConfirmModal();
        }
        
        // 页面加载时初始化数据
        document.addEventListener('DOMContentLoaded', function() {
            loadMerchants();
            loadCategoriesForModal();
            loadFloorsForModal();
        });
        
        // 加载商户数据
        function loadMerchants() {
            fetch('index.php?action=get_merchants')
                .then(response => {
                    if (!response.ok) {
                        throw new Error(`HTTP ${response.status}: ${response.statusText}`);
                    }
                    return response.text();
                })
                .then(text => {
                    try {
                        const data = JSON.parse(text);
                        if (data.success && data.merchants) {
                            allMerchants = data.merchants; // 保存所有商户数据
                            renderMerchantList(data.merchants);
                            loadFloorsForFilter(); // 加载楼层筛选选项
                        } else {
                            allMerchants = [];
                            renderMerchantList([]);
                            showMessage('暂无商户数据', 'info');
                        }
                    } catch (e) {
                        console.error('JSON解析错误:', e);
                        console.error('服务器响应:', text);
                        allMerchants = [];
                        renderMerchantList([]);
                        showMessage('加载商户数据失败，服务器响应格式错误', 'error');
                    }
                })
                .catch(error => {
                    console.error('网络错误:', error);
                    allMerchants = [];
                    renderMerchantList([]);
                    showMessage(`加载商户数据失败: ${error.message}`, 'error');
                });
        }
        
        // 渲染商户列表
        function renderMerchantList(merchants) {
            const container = document.getElementById('merchantList');
            const selectedFloorId = document.getElementById('floorFilter').value;
            
            if (merchants.length === 0) {
                const emptyMessage = selectedFloorId ? '该楼层暂无商户数据' : '暂无商户数据';
                container.innerHTML = `
                    <div class="empty-state">
                        <img src="assets/empty-data.svg" alt="暂无数据">
                        <p>${emptyMessage}</p>
                    </div>
                `;
                return;
            }
            
            // 如果选择了特定楼层，直接显示该楼层的商户
            if (selectedFloorId) {
                let html = '';
                const selectedFloor = allFloors.find(floor => floor.id === selectedFloorId);
                const floorName = selectedFloor ? selectedFloor.name : '未知楼层';
                
                html += `
                    <div class="floor-group">
                        <h3 class="floor-title">${floorName} (${merchants.length}个商户)</h3>
                        <div class="floor-merchants">
                `;
                
                merchants.forEach(merchant => {
                    let itemInfo = `
                        <div class="merchant-item-name">${merchant.name}</div>
                        <div class="merchant-item-time">创建时间: ${merchant.created_at}</div>
                    `;
                    
                    if (merchant.category_name) {
                        itemInfo += `<div class="merchant-item-detail">分类: ${merchant.category_name}</div>`;
                    }
                    
                    html += `
                        <div class="merchant-item">
                            <div class="merchant-item-info">
                                ${itemInfo}
                            </div>
                            <div class="merchant-item-actions">
                                <button class="edit-btn" onclick="editMerchant('${merchant.id}', '${merchant.name}', '${merchant.category || ''}', '${merchant.floor || ''}')">编辑</button>
                                <button class="delete-btn" onclick="deleteMerchant('${merchant.id}', '${merchant.name}')">删除</button>
                            </div>
                        </div>
                    `;
                });
                
                html += `
                        </div>
                    </div>
                `;
                
                container.innerHTML = html;
                return;
            }
            
            // 显示所有楼层时，按楼层分组
            const merchantsByFloor = {};
            merchants.forEach(merchant => {
                const floorName = merchant.floor_name || '未分配楼层';
                if (!merchantsByFloor[floorName]) {
                    merchantsByFloor[floorName] = [];
                }
                merchantsByFloor[floorName].push(merchant);
            });
            
            // 渲染分组后的商户列表
            let html = '';
            Object.keys(merchantsByFloor).sort().forEach(floorName => {
                const floorMerchants = merchantsByFloor[floorName];
                html += `
                    <div class="floor-group">
                        <h3 class="floor-title">${floorName} (${floorMerchants.length}个商户)</h3>
                        <div class="floor-merchants">
                `;
                
                floorMerchants.forEach(merchant => {
                    let itemInfo = `
                        <div class="merchant-item-name">${merchant.name}</div>
                        <div class="merchant-item-time">创建时间: ${merchant.created_at}</div>
                    `;
                    
                    if (merchant.category_name) {
                        itemInfo += `<div class="merchant-item-detail">分类: ${merchant.category_name}</div>`;
                    }
                    
                    html += `
                        <div class="merchant-item">
                            <div class="merchant-item-info">
                                ${itemInfo}
                            </div>
                            <div class="merchant-item-actions">
                                <button class="edit-btn" onclick="editMerchant('${merchant.id}', '${merchant.name}', '${merchant.category || ''}', '${merchant.floor || ''}')">编辑</button>
                                <button class="delete-btn" onclick="deleteMerchant('${merchant.id}', '${merchant.name}')">删除</button>
                            </div>
                        </div>
                    `;
                });
                
                html += `
                        </div>
                    </div>
                `;
            });
            
            container.innerHTML = html;
        }
        
        // 打开添加模态框
        function openAddModal() {
            currentId = '';
            isEditMode = false;
            document.getElementById('modalTitle').textContent = '添加商户';
            document.getElementById('merchantName').value = '';
            document.getElementById('merchantCategory').value = '';
            document.getElementById('merchantFloor').value = '';
            document.getElementById('merchantModal').style.display = 'block';
        }
        
        // 编辑商户
        function editMerchant(id, name, categoryId, floorId) {
            currentId = id;
            isEditMode = true;
            document.getElementById('modalTitle').textContent = '编辑商户';
            document.getElementById('merchantName').value = name;
            document.getElementById('merchantCategory').value = categoryId;
            document.getElementById('merchantFloor').value = floorId;
            document.getElementById('merchantModal').style.display = 'block';
        }
        
        // 关闭模态框
        function closeModal() {
            document.getElementById('merchantModal').style.display = 'none';
        }
        
        // 保存商户
        function saveMerchant() {
            const name = document.getElementById('merchantName').value.trim();
            const category = document.getElementById('merchantCategory').value;
            const floor = document.getElementById('merchantFloor').value;
            
            if (!name) {
                alert('请输入商户名称');
                return;
            }
            
            if (!category) {
                alert('请选择分类');
                return;
            }
            
            if (!floor) {
                alert('请选择楼层');
                return;
            }
            
            const formData = new FormData();
            formData.append('name', name);
            formData.append('category', category);
            formData.append('floor', floor);
            
            if (isEditMode) {
                formData.append('action', 'edit_merchant');
                formData.append('id', currentId);
            } else {
                formData.append('action', 'add_merchant');
            }
            
            fetch('index.php', {
                method: 'POST',
                body: formData
            })
            .then(response => {
                if (!response.ok) {
                    throw new Error(`HTTP ${response.status}: ${response.statusText}`);
                }
                return response.text();
            })
            .then(text => {
                try {
                    const data = JSON.parse(text);
                    if (data.success) {
                        closeModal();
                        loadMerchants();
                        showMessage(isEditMode ? '商户更新成功！' : '商户添加成功！', 'success');
                    } else {
                        showMessage(data.message || '操作失败', 'error');
                    }
                } catch (e) {
                    console.error('JSON解析错误:', e);
                    console.error('服务器响应:', text);
                    showMessage('服务器响应格式错误，请检查后端配置', 'error');
                }
            })
            .catch(error => {
                console.error('网络错误:', error);
                showMessage(`网络请求失败: ${error.message}`, 'error');
            });
        }
        
        // 删除商户
        function deleteMerchant(id, name) {
            showConfirm(`确定要删除商户"${name}"吗？`, function() {
                const formData = new FormData();
                formData.append('action', 'delete_merchant');
                formData.append('id', id);
                
                fetch('index.php', {
                    method: 'POST',
                    body: formData
                })
                    .then(response => {
                        if (!response.ok) {
                            throw new Error(`HTTP ${response.status}: ${response.statusText}`);
                        }
                        return response.text();
                    })
                    .then(text => {
                        try {
                            const data = JSON.parse(text);
                            if (data.success) {
                                loadMerchants();
                                showMessage('商户删除成功！', 'success');
                            } else {
                                showMessage(data.message || '删除失败', 'error');
                            }
                        } catch (e) {
                            console.error('JSON解析错误:', e);
                            console.error('服务器响应:', text);
                            showMessage('服务器响应格式错误，请检查后端配置', 'error');
                        }
                    })
                    .catch(error => {
                        console.error('网络错误:', error);
                        showMessage(`网络请求失败: ${error.message}`, 'error');
                    });
            });
        }
        
        // 加载分类选项
        function loadCategoriesForModal() {
            fetch('index.php?action=get_categories')
                .then(response => {
                    if (!response.ok) {
                        throw new Error(`HTTP ${response.status}: ${response.statusText}`);
                    }
                    return response.text();
                })
                .then(text => {
                    try {
                        const data = JSON.parse(text);
                        const select = document.getElementById('merchantCategory');
                        select.innerHTML = '<option value="">请选择分类</option>';
                        
                        if (data.success && data.categories) {
                            data.categories.forEach(category => {
                                select.innerHTML += `<option value="${category.id}">${category.name}</option>`;
                            });
                        } else {
                            console.error('加载分类失败:', data.message || '未知错误');
                            showMessage('加载分类数据失败', 'error');
                        }
                    } catch (e) {
                        console.error('JSON解析错误:', e);
                        console.error('服务器响应:', text);
                        showMessage('加载分类数据失败，服务器响应格式错误', 'error');
                    }
                })
                .catch(error => {
                    console.error('网络错误:', error);
                    showMessage(`加载分类数据失败: ${error.message}`, 'error');
                });
        }
        
        // 加载楼层选项
        function loadFloorsForModal() {
            fetch('index.php?action=get_floor_config')
                .then(response => {
                    if (!response.ok) {
                        throw new Error(`HTTP ${response.status}: ${response.statusText}`);
                    }
                    return response.text();
                })
                .then(text => {
                    try {
                        const data = JSON.parse(text);
                        const select = document.getElementById('merchantFloor');
                        select.innerHTML = '<option value="">请选择楼层</option>';
                        
                        if (data.success && data.floors) {
                            data.floors.forEach(floor => {
                                select.innerHTML += `<option value="${floor.id}">${floor.name}</option>`;
                            });
                        } else {
                            console.error('加载楼层失败:', data.message || '未知错误');
                            showMessage('加载楼层数据失败', 'error');
                        }
                    } catch (e) {
                        console.error('JSON解析错误:', e);
                        console.error('服务器响应:', text);
                        showMessage('加载楼层数据失败，服务器响应格式错误', 'error');
                    }
                })
                .catch(error => {
                    console.error('网络错误:', error);
                    showMessage(`加载楼层数据失败: ${error.message}`, 'error');
                });
        }
        
        // 加载楼层筛选选项
        function loadFloorsForFilter() {
            fetch('index.php?action=get_floor_config')
                .then(response => {
                    if (!response.ok) {
                        throw new Error(`HTTP ${response.status}: ${response.statusText}`);
                    }
                    return response.text();
                })
                .then(text => {
                    try {
                        const data = JSON.parse(text);
                        const select = document.getElementById('floorFilter');
                        select.innerHTML = '<option value="">全部楼层</option>';
                        
                        if (data.success && data.floors) {
                            allFloors = data.floors; // 保存楼层数据
                            data.floors.forEach(floor => {
                                select.innerHTML += `<option value="${floor.id}">${floor.name}</option>`;
                            });
                        }
                    } catch (e) {
                        console.error('加载楼层筛选选项失败:', e);
                    }
                })
                .catch(error => {
                    console.error('加载楼层筛选选项失败:', error);
                });
        }
        
        // 按楼层筛选商户
        function filterByFloor() {
            const selectedFloorId = document.getElementById('floorFilter').value;
            
            if (!selectedFloorId) {
                // 显示所有商户
                renderMerchantList(allMerchants);
            } else {
                // 筛选指定楼层的商户
                const filteredMerchants = allMerchants.filter(merchant => merchant.floor === selectedFloorId);
                renderMerchantList(filteredMerchants);
            }
        }
        
        // 点击模态框外部关闭
        window.onclick = function(event) {
            const modal = document.getElementById('merchantModal');
            if (event.target === modal) {
                closeModal();
            }
        }
        
        // 回车键保存
        document.addEventListener('keydown', function(event) {
            if (event.key === 'Enter' && document.getElementById('merchantModal').style.display === 'block') {
                saveMerchant();
            }
        });
    </script>
</body>
</html>