<?php

// 检查登录状态
function handleCheckLogin() {
    $logged_in = isset($_SESSION['admin']);
    $username = $logged_in ? $_SESSION['admin'] : '';
    echo json_encode(['success' => true, 'logged_in' => $logged_in, 'username' => $username]);
}

// 登录处理
function handleLogin() {
    $username = $_POST['username'] ?? '';
    $password = $_POST['password'] ?? '';
    
    if (empty($username) || empty($password)) {
        echo json_encode(['success' => false, 'message' => '用户名和密码不能为空']);
        return;
    }
    
    $admins = json_decode(file_get_contents(ADMIN_FILE), true);
    
    if (isset($admins[$username]) && password_verify($password, $admins[$username]['password'])) {
        $_SESSION['admin'] = $username;
        echo json_encode(['success' => true, 'message' => '登录成功']);
    } else {
        echo json_encode(['success' => false, 'message' => '用户名或密码错误']);
    }
}

// 退出登录
function handleLogout() {
    session_destroy();
    echo json_encode(['success' => true, 'message' => '已退出登录']);
}

// 获取楼层列表
function handleGetFloors() {
    $floorFile = DATA_DIR . '/floor_config.json';
    $floors = [];
    
    if (file_exists($floorFile)) {
        $floors = json_decode(file_get_contents($floorFile), true) ?: [];
    }
    
    echo json_encode(['success' => true, 'floors' => $floors]);
}

// 获取账号数据
function handleGetAccounts() {
    $floor = $_GET['floor'] ?? 'all';
    $accounts = [];
    
    if ($floor === 'all') {
        // 获取所有楼层数据
        $dataDir = DATA_DIR;
        if (is_dir($dataDir)) {
            $files = scandir($dataDir);
            foreach ($files as $file) {
                if (preg_match('/^floor_(.+)\.json$/', $file) && $file !== 'floor_config.json') {
                    $floorData = json_decode(file_get_contents($dataDir . '/' . $file), true);
                    if ($floorData) {
                        // 过滤掉楼层配置数据，只保留真正的账户数据
                        $validAccounts = array_filter($floorData, function($item) {
                            // 楼层配置数据只有 id, name, created_at, status, updated_at, operator 字段
                            // 账户数据会有 merchant_name, account, phone 等字段
                            return isset($item['merchant_name']) || isset($item['account']) || isset($item['phone']);
                        });
                        $accounts = array_merge($accounts, $validAccounts);
                    }
                }
            }
        }
    } else {
        // 获取指定楼层数据
        $floorFile = DATA_DIR . '/floor_' . $floor . '.json';
        if (file_exists($floorFile)) {
            $floorData = json_decode(file_get_contents($floorFile), true) ?: [];
            // 过滤掉楼层配置数据，只保留真正的账户数据
            $accounts = array_filter($floorData, function($item) {
                return isset($item['merchant_name']) || isset($item['account']) || isset($item['phone']);
            });
        }
    }
    
    // 加载楼层和分类配置
    $floorConfig = [];
    $categoryConfig = [];
    
    $floorConfigFile = DATA_DIR . '/floor_config.json';
    if (file_exists($floorConfigFile)) {
        $floors = json_decode(file_get_contents($floorConfigFile), true);
        foreach ($floors as $floorItem) {
            $floorConfig[$floorItem['id']] = $floorItem['name'];
        }
    }
    
    $categoryConfigFile = DATA_DIR . '/categories.json';
    if (file_exists($categoryConfigFile)) {
        $categories = json_decode(file_get_contents($categoryConfigFile), true);
        foreach ($categories as $categoryItem) {
            $categoryConfig[$categoryItem['id']] = $categoryItem['name'];
        }
    }
    
    // 转换ID为名称
    foreach ($accounts as &$account) {
        // 转换楼层ID为名称
        if (isset($account['floor']) && isset($floorConfig[$account['floor']])) {
            $account['floor_name'] = $floorConfig[$account['floor']];
        } else {
            $account['floor_name'] = $account['floor'] ?? '-';
        }
        
        // 转换分类ID为名称
        if (isset($account['merchant_category']) && isset($categoryConfig[$account['merchant_category']])) {
            $account['merchant_category_name'] = $categoryConfig[$account['merchant_category']];
        } else {
            $account['merchant_category_name'] = $account['merchant_category'] ?? '-';
        }
        
        // 确保status字段存在，如果不存在则设为active
        if (!isset($account['status'])) {
            $account['status'] = 'active';
        }
    }
    
    // 自定义排序：冻结账号显示在最后，其他按更新时间倒序
    usort($accounts, function($a, $b) {
        // 如果一个是冻结状态，一个不是，冻结的排在后面
        if ($a['status'] === 'frozen' && $b['status'] !== 'frozen') {
            return 1;
        }
        if ($a['status'] !== 'frozen' && $b['status'] === 'frozen') {
            return -1;
        }
        
        // 如果都是冻结状态或都不是冻结状态，按更新时间倒序排列
        return strtotime($b['updated_at']) - strtotime($a['updated_at']);
    });
    
    echo json_encode(['success' => true, 'accounts' => $accounts]);
}

// 搜索处理
function handleSearch() {
    $type = $_GET['type'] ?? '';
    $value = $_GET['value'] ?? '';
    $floor = $_GET['floor'] ?? '';
    
    // 至少需要一个搜索条件
    if (empty($type) && empty($floor)) {
        echo json_encode(['success' => false, 'message' => '搜索参数不能为空']);
        return;
    }
    
    $accounts = [];
    $filteredAccounts = [];
    
    // 获取所有楼层数据
    $dataDir = DATA_DIR;
    if (is_dir($dataDir)) {
        $files = scandir($dataDir);
        foreach ($files as $file) {
            if (preg_match('/^floor_(.+)\.json$/', $file)) {
                $floorData = json_decode(file_get_contents($dataDir . '/' . $file), true);
                if ($floorData) {
                    // 过滤掉楼层配置数据，只保留真正的账户数据
                    $accountsInFloor = array_filter($floorData, function($item) {
                        return isset($item['merchant_name']) || isset($item['account']) || isset($item['phone']);
                    });
                    $accounts = array_merge($accounts, $accountsInFloor);
                }
            }
        }
    }
    
    // 加载楼层和分类配置
    $floorConfig = [];
    $categoryConfig = [];
    $merchantConfig = [];
    
    $floorConfigFile = DATA_DIR . '/floor_config.json';
    if (file_exists($floorConfigFile)) {
        $floors = json_decode(file_get_contents($floorConfigFile), true);
        foreach ($floors as $floorItem) {
            $floorConfig[$floorItem['id']] = $floorItem['name'];
        }
    }
    
    $categoryConfigFile = DATA_DIR . '/categories.json';
    if (file_exists($categoryConfigFile)) {
        $categories = json_decode(file_get_contents($categoryConfigFile), true);
        foreach ($categories as $categoryItem) {
            $categoryConfig[$categoryItem['id']] = $categoryItem['name'];
        }
    }
    
    $merchantConfigFile = DATA_DIR . '/merchants.json';
    if (file_exists($merchantConfigFile)) {
        $merchants = json_decode(file_get_contents($merchantConfigFile), true);
        foreach ($merchants as $merchantItem) {
            $merchantConfig[$merchantItem['id']] = $merchantItem['name'];
        }
    }
    
    // 过滤账号数据
    foreach ($accounts as $account) {
        // 转换楼层ID为名称
        if (isset($account['floor']) && isset($floorConfig[$account['floor']])) {
            $account['floor_name'] = $floorConfig[$account['floor']];
        } else {
            $account['floor_name'] = $account['floor'] ?? '-';
        }
        
        // 转换分类ID为名称
        if (isset($account['merchant_category']) && isset($categoryConfig[$account['merchant_category']])) {
            $account['merchant_category_name'] = $categoryConfig[$account['merchant_category']];
        } else {
            $account['merchant_category_name'] = $account['merchant_category'] ?? '-';
        }
        
        // 商户名称直接从账号数据中获取，不需要转换
        $account['merchant_name'] = $account['merchant_name'] ?? '-';
        
        // 确保status字段存在，如果不存在则设为active
        if (!isset($account['status'])) {
            $account['status'] = 'active';
        }
        
        // 根据搜索条件进行过滤
        $match = true;
        
        // 如果指定了搜索类型和值，进行匹配
        if (!empty($type) && !empty($value)) {
            $typeMatch = false;
            switch ($type) {
                case 'merchant':
                    // 按商户搜索 - 支持模糊匹配
                    if (isset($account['merchant_name'])) {
                        $typeMatch = stripos($account['merchant_name'], $value) !== false;
                    }
                    break;
            }
            $match = $match && $typeMatch;
        }
        
        // 如果指定了楼层筛选，进行匹配
        if (!empty($floor)) {
            $floorMatch = ($account['floor'] == $floor);
            $match = $match && $floorMatch;
        }
        
        if ($match) {
            $filteredAccounts[] = $account;
        }
    }
    
    echo json_encode(['success' => true, 'accounts' => $filteredAccounts]);
}

// 新增账号
function handleAdd() {
    if (!isset($_SESSION['admin'])) {
        echo json_encode(['success' => false, 'message' => '请先登录']);
        return;
    }
    
    $merchantName = $_POST['merchant_name'] ?? '';
    $merchantCategory = $_POST['merchant_category'] ?? '';
    $account = $_POST['account'] ?? '';
    $phone = $_POST['phone'] ?? '';
    $floor = $_POST['floor'] ?? '';
    $remark = $_POST['remark'] ?? '';
    $editId = $_POST['edit_id'] ?? '';
    
    if (empty($merchantName) || empty($merchantCategory) || empty($account) || empty($phone) || empty($floor)) {
        echo json_encode(['success' => false, 'message' => '请填写所有必填字段']);
        return;
    }
    
    // 如果是编辑模式
    if (!empty($editId)) {
        // 编辑现有账号
        $dataDir = DATA_DIR;
        if (is_dir($dataDir)) {
            $files = scandir($dataDir);
            foreach ($files as $file) {
                if (preg_match('/^floor_(.+)\.json$/', $file)) {
                    $floorFile = $dataDir . '/' . $file;
                    $accounts = json_decode(file_get_contents($floorFile), true);
                    
                    if ($accounts) {
                        foreach ($accounts as &$existingAccount) {
                            if ($existingAccount['id'] === $editId) {
                                // 检查账号是否被其他记录使用（排除当前记录）
                                if ($existingAccount['account'] !== $account && accountExists($account)) {
                                    echo json_encode(['success' => false, 'message' => '账号已存在']);
                                    return;
                                }
                                
                                // 更新账号信息
                                $existingAccount['merchant_name'] = $merchantName;
                                $existingAccount['merchant_category'] = $merchantCategory;
                                $existingAccount['account'] = $account;
                                $existingAccount['phone'] = $phone;
                                $existingAccount['floor'] = $floor;
                                $existingAccount['remark'] = $remark;
                                $existingAccount['updated_at'] = date('Y-m-d H:i:s');
                                $existingAccount['operator'] = $_SESSION['admin'];
                                
                                if (file_put_contents($floorFile, json_encode($accounts, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE))) {
                                    echo json_encode(['success' => true, 'message' => '账号更新成功']);
                                } else {
                                    echo json_encode(['success' => false, 'message' => '保存失败']);
                                }
                                return;
                            }
                        }
                    }
                }
            }
        }
        echo json_encode(['success' => false, 'message' => '账号不存在']);
        return;
    }
    
    // 新增模式
    // 检查账号是否已存在
    if (accountExists($account)) {
        echo json_encode(['success' => false, 'message' => '账号已存在']);
        return;
    }
    
    $floorFile = DATA_DIR . '/floor_' . $floor . '.json';
    $accounts = [];
    
    if (file_exists($floorFile)) {
        $accounts = json_decode(file_get_contents($floorFile), true) ?: [];
    }
    
    $newAccount = [
        'id' => uniqid(),
        'merchant_name' => $merchantName,
        'merchant_category' => $merchantCategory,
        'account' => $account,
        'phone' => $phone,
        'floor' => $floor,
        'remark' => $remark,
        'status' => 'active',
        'created_at' => date('Y-m-d H:i:s'),
        'updated_at' => date('Y-m-d H:i:s'),
        'operator' => $_SESSION['admin']
    ];
    
    $accounts[] = $newAccount;
    
    if (file_put_contents($floorFile, json_encode($accounts, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE))) {
        echo json_encode(['success' => true, 'message' => '账号添加成功']);
    } else {
        echo json_encode(['success' => false, 'message' => '保存失败']);
    }
}

// 删除（冻结）账号
function handleDelete() {
    if (!isset($_SESSION['admin'])) {
        echo json_encode(['success' => false, 'message' => '请先登录']);
        return;
    }
    
    $accountId = $_POST['id'] ?? '';
    
    if (empty($accountId)) {
        echo json_encode(['success' => false, 'message' => '账号ID不能为空']);
        return;
    }
    
    // 查找并更新账号状态
    $dataDir = DATA_DIR;
    if (is_dir($dataDir)) {
        $files = scandir($dataDir);
        foreach ($files as $file) {
            if (preg_match('/^floor_(.+)\.json$/', $file)) {
                $floorFile = $dataDir . '/' . $file;
                $accounts = json_decode(file_get_contents($floorFile), true);
                
                if ($accounts) {
                    foreach ($accounts as &$account) {
                        if ($account['id'] === $accountId) {
                            $account['status'] = 'frozen';
                            $account['updated_at'] = date('Y-m-d H:i:s');
                            $account['operator'] = $_SESSION['admin'];
                            
                            if (file_put_contents($floorFile, json_encode($accounts, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE))) {
                                echo json_encode(['success' => true, 'message' => '账号已冻结']);
                            } else {
                                echo json_encode(['success' => false, 'message' => '操作失败']);
                            }
                            return;
                        }
                    }
                }
            }
        }
    }
    
    echo json_encode(['success' => false, 'message' => '账号不存在']);
}

// 永久删除账号
function handlePermanentDelete() {
    if (!isset($_SESSION['admin'])) {
        echo json_encode(['success' => false, 'message' => '请先登录']);
        return;
    }
    
    $accountId = $_POST['id'] ?? '';
    
    if (empty($accountId)) {
        echo json_encode(['success' => false, 'message' => '账号ID不能为空']);
        return;
    }
    
    // 查找并删除账号
    $dataDir = DATA_DIR;
    if (is_dir($dataDir)) {
        $files = scandir($dataDir);
        foreach ($files as $file) {
            if (preg_match('/^floor_(.+)\.json$/', $file)) {
                $floorFile = $dataDir . '/' . $file;
                $accounts = json_decode(file_get_contents($floorFile), true);
                
                if ($accounts) {
                    foreach ($accounts as $index => $account) {
                        if ($account['id'] === $accountId) {
                            // 检查账号是否已冻结
                            if ($account['status'] !== 'frozen') {
                                echo json_encode(['success' => false, 'message' => '只能删除已冻结的账号']);
                                return;
                            }
                            
                            // 检查是否为测试账号（账号名包含"test"或"测试"）
                            $isTestAccount = false;
                            if (isset($account['account'])) {
                                $accountName = strtolower($account['account']);
                                if (strpos($accountName, 'test') !== false || 
                                    strpos($account['account'], '测试') !== false) {
                                    $isTestAccount = true;
                                }
                            }
                            
                            // 如果不是测试账号，则不允许删除冻结账号
                            if (!$isTestAccount) {
                                echo json_encode(['success' => false, 'message' => '冻结账号不能被删除，只有测试账号可以删除']);
                                return;
                            }
                            
                            // 从数组中移除账号
                            array_splice($accounts, $index, 1);
                            
                            if (file_put_contents($floorFile, json_encode($accounts, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE))) {
                                echo json_encode(['success' => true, 'message' => '测试账号已永久删除']);
                            } else {
                                echo json_encode(['success' => false, 'message' => '删除失败']);
                            }
                            return;
                        }
                    }
                }
            }
        }
    }
    
    echo json_encode(['success' => false, 'message' => '账号不存在']);
}

// 导出Excel
function handleExport() {
    if (!isset($_SESSION['admin'])) {
        echo json_encode(['success' => false, 'message' => '请先登录']);
        return;
    }
    
    // 获取所有账号数据
    $accounts = [];
    $dataDir = DATA_DIR;
    
    if (is_dir($dataDir)) {
        $files = scandir($dataDir);
        foreach ($files as $file) {
            if (preg_match('/^floor_(.+)\.json$/', $file)) {
                $floorData = json_decode(file_get_contents($dataDir . '/' . $file), true);
                if ($floorData) {
                    $accounts = array_merge($accounts, $floorData);
                }
            }
        }
    }
    
    // 按楼层和创建时间排序
    usort($accounts, function($a, $b) {
        $floorCompare = strcmp($a['floor'], $b['floor']);
        if ($floorCompare === 0) {
            return strtotime($a['created_at']) - strtotime($b['created_at']);
        }
        return $floorCompare;
    });
    
    // 生成CSV格式（Excel可以打开）
    $filename = '美团子账号数据_' . date('Y-m-d_H-i-s') . '.csv';
    
    header('Content-Type: text/csv; charset=UTF-8');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    header('Cache-Control: must-revalidate, post-check=0, pre-check=0');
    header('Expires: 0');
    
    // 输出BOM以支持中文
    echo "\xEF\xBB\xBF";
    
    $output = fopen('php://output', 'w');
    
    // 写入表头
    fputcsv($output, ['商户名称', '账号', '手机号', '楼层', '备注', '状态', '创建时间', '更新时间', '操作员']);
    
    // 写入数据
    foreach ($accounts as $account) {
        $status = $account['status'] === 'active' ? '正常' : '已冻结';
        fputcsv($output, [
            $account['merchant_name'],
            $account['account'],
            $account['phone'],
            $account['floor'],
            $account['remark'],
            $status,
            $account['created_at'],
            $account['updated_at'],
            $account['operator']
        ]);
    }
    
    fclose($output);
    exit;
}

// 修改密码
function handleChangePassword() {
    if (!isset($_SESSION['admin'])) {
        echo json_encode(['success' => false, 'message' => '请先登录']);
        return;
    }
    
    $newPassword = $_POST['new_password'] ?? '';
    
    if (empty($newPassword)) {
        echo json_encode(['success' => false, 'message' => '新密码不能为空']);
        return;
    }
    
    $admins = json_decode(file_get_contents(ADMIN_FILE), true);
    $admins[$_SESSION['admin']]['password'] = password_hash($newPassword, PASSWORD_DEFAULT);
    
    if (file_put_contents(ADMIN_FILE, json_encode($admins, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE))) {
        echo json_encode(['success' => true, 'message' => '密码修改成功']);
    } else {
        echo json_encode(['success' => false, 'message' => '密码修改失败']);
    }
}

// 添加管理员
function handleAddAdmin() {
    if (!isset($_SESSION['admin'])) {
        echo json_encode(['success' => false, 'message' => '请先登录']);
        return;
    }
    
    $username = $_POST['username'] ?? '';
    $password = $_POST['password'] ?? '';
    
    if (empty($username) || empty($password)) {
        echo json_encode(['success' => false, 'message' => '用户名和密码不能为空']);
        return;
    }
    
    $admins = json_decode(file_get_contents(ADMIN_FILE), true);
    
    if (isset($admins[$username])) {
        echo json_encode(['success' => false, 'message' => '管理员已存在']);
        return;
    }
    
    $admins[$username] = [
        'password' => password_hash($password, PASSWORD_DEFAULT),
        'created_at' => date('Y-m-d H:i:s'),
        'created_by' => $_SESSION['admin']
    ];
    
    if (file_put_contents(ADMIN_FILE, json_encode($admins, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE))) {
        echo json_encode(['success' => true, 'message' => '管理员添加成功']);
    } else {
        echo json_encode(['success' => false, 'message' => '管理员添加失败']);
    }
}

// 检查账号是否存在
function accountExists($account) {
    $dataDir = DATA_DIR;
    
    if (is_dir($dataDir)) {
        $files = scandir($dataDir);
        foreach ($files as $file) {
            if (preg_match('/^floor_(.+)\.json$/', $file)) {
                $floorData = json_decode(file_get_contents($dataDir . '/' . $file), true);
                if ($floorData) {
                    foreach ($floorData as $existingAccount) {
                        if ($existingAccount['account'] === $account) {
                            return true;
                        }
                    }
                }
            }
        }
    }
    
    return false;
}

// 基础数据管理API

// 获取商户名称列表
function handleGetMerchants() {
    $merchantFile = DATA_DIR . '/merchants.json';
    $categoryFile = DATA_DIR . '/categories.json';
    $floorFile = DATA_DIR . '/floor_config.json';
    
    $merchants = [];
    $categories = [];
    $floors = [];
    
    if (file_exists($merchantFile)) {
        $merchants = json_decode(file_get_contents($merchantFile), true) ?: [];
    }
    
    if (file_exists($categoryFile)) {
        $categories = json_decode(file_get_contents($categoryFile), true) ?: [];
    }
    
    if (file_exists($floorFile)) {
        $floors = json_decode(file_get_contents($floorFile), true) ?: [];
    }
    
    // 创建分类和楼层的映射表
    $categoryMap = [];
    foreach ($categories as $category) {
        $categoryMap[$category['id']] = $category['name'];
    }
    
    $floorMap = [];
    foreach ($floors as $floor) {
        $floorMap[$floor['id']] = $floor['name'];
    }
    
    // 为商户添加分类名称和楼层名称
    foreach ($merchants as &$merchant) {
        if (isset($merchant['category']) && isset($categoryMap[$merchant['category']])) {
            $merchant['category_name'] = $categoryMap[$merchant['category']];
        }
        if (isset($merchant['floor']) && isset($floorMap[$merchant['floor']])) {
            $merchant['floor_name'] = $floorMap[$merchant['floor']];
        }
    }
    
    echo json_encode(['success' => true, 'merchants' => $merchants]);
}

// 获取商户分类列表
function handleGetCategories() {
    $categoryFile = DATA_DIR . '/categories.json';
    $categories = [];
    
    if (file_exists($categoryFile)) {
        $categories = json_decode(file_get_contents($categoryFile), true) ?: [];
    }
    
    echo json_encode(['success' => true, 'categories' => $categories]);
}

// 获取楼层配置列表
function handleGetFloorConfig() {
    $floorFile = DATA_DIR . '/floor_config.json';
    $floors = [];
    
    if (file_exists($floorFile)) {
        $floors = json_decode(file_get_contents($floorFile), true) ?: [];
    }
    
    echo json_encode(['success' => true, 'floors' => $floors]);
}

// 添加商户名称
function handleAddMerchant() {
    if (!isset($_SESSION['admin'])) {
        echo json_encode(['success' => false, 'message' => '请先登录']);
        return;
    }
    
    $name = trim($_POST['name'] ?? '');
    $category = trim($_POST['category'] ?? '');
    $floor = trim($_POST['floor'] ?? '');
    
    if (empty($name)) {
        echo json_encode(['success' => false, 'message' => '商户名称不能为空']);
        return;
    }
    
    $merchantFile = DATA_DIR . '/merchants.json';
    $merchants = [];
    
    if (file_exists($merchantFile)) {
        $merchants = json_decode(file_get_contents($merchantFile), true) ?: [];
    }
    
    // 检查是否已存在
    foreach ($merchants as $merchant) {
        if ($merchant['name'] === $name) {
            echo json_encode(['success' => false, 'message' => '商户名称已存在']);
            return;
        }
    }
    
    $newMerchant = [
        'id' => uniqid(),
        'name' => $name,
        'created_at' => date('Y-m-d H:i:s')
    ];
    
    // 如果提供了分类和楼层信息，则保存
    if (!empty($category)) {
        $newMerchant['category'] = $category;
    }
    if (!empty($floor)) {
        $newMerchant['floor'] = $floor;
    }
    
    $merchants[] = $newMerchant;
    
    file_put_contents($merchantFile, json_encode($merchants, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    echo json_encode(['success' => true, 'message' => '添加成功']);
}

// 添加商户分类
function handleAddCategory() {
    if (!isset($_SESSION['admin'])) {
        echo json_encode(['success' => false, 'message' => '请先登录']);
        return;
    }
    
    $name = trim($_POST['name'] ?? '');
    
    if (empty($name)) {
        echo json_encode(['success' => false, 'message' => '分类名称不能为空']);
        return;
    }
    
    $categoryFile = DATA_DIR . '/categories.json';
    $categories = [];
    
    if (file_exists($categoryFile)) {
        $categories = json_decode(file_get_contents($categoryFile), true) ?: [];
    }
    
    // 检查是否已存在
    foreach ($categories as $category) {
        if ($category['name'] === $name) {
            echo json_encode(['success' => false, 'message' => '分类名称已存在']);
            return;
        }
    }
    
    $categories[] = [
        'id' => uniqid(),
        'name' => $name,
        'created_at' => date('Y-m-d H:i:s')
    ];
    
    file_put_contents($categoryFile, json_encode($categories, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    echo json_encode(['success' => true, 'message' => '添加成功']);
}

// 添加楼层配置
function handleAddFloorConfig() {
    if (!isset($_SESSION['admin'])) {
        echo json_encode(['success' => false, 'message' => '请先登录']);
        return;
    }
    
    $name = trim($_POST['name'] ?? '');
    
    if (empty($name)) {
        echo json_encode(['success' => false, 'message' => '楼层名称不能为空']);
        return;
    }
    
    $floorFile = DATA_DIR . '/floor_config.json';
    $floors = [];
    
    if (file_exists($floorFile)) {
        $floors = json_decode(file_get_contents($floorFile), true) ?: [];
    }
    
    // 检查是否已存在
    foreach ($floors as $floor) {
        if ($floor['name'] === $name) {
            echo json_encode(['success' => false, 'message' => '楼层名称已存在']);
            return;
        }
    }
    
    $floors[] = [
        'id' => uniqid(),
        'name' => $name,
        'created_at' => date('Y-m-d H:i:s')
    ];
    
    file_put_contents($floorFile, json_encode($floors, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    echo json_encode(['success' => true, 'message' => '添加成功']);
}

// 删除商户名称
function handleDeleteMerchant() {
    if (!isset($_SESSION['admin'])) {
        echo json_encode(['success' => false, 'message' => '请先登录']);
        return;
    }
    
    $id = $_POST['id'] ?? '';
    
    if (empty($id)) {
        echo json_encode(['success' => false, 'message' => 'ID不能为空']);
        return;
    }
    
    $merchantFile = DATA_DIR . '/merchants.json';
    $merchants = [];
    
    if (file_exists($merchantFile)) {
        $merchants = json_decode(file_get_contents($merchantFile), true) ?: [];
    }
    
    $merchants = array_filter($merchants, function($merchant) use ($id) {
        return $merchant['id'] !== $id;
    });
    
    file_put_contents($merchantFile, json_encode(array_values($merchants), JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    echo json_encode(['success' => true, 'message' => '删除成功']);
}

// 删除商户分类
function handleDeleteCategory() {
    if (!isset($_SESSION['admin'])) {
        echo json_encode(['success' => false, 'message' => '请先登录']);
        return;
    }
    
    $id = $_POST['id'] ?? '';
    
    if (empty($id)) {
        echo json_encode(['success' => false, 'message' => 'ID不能为空']);
        return;
    }
    
    $categoryFile = DATA_DIR . '/categories.json';
    $categories = [];
    
    if (file_exists($categoryFile)) {
        $categories = json_decode(file_get_contents($categoryFile), true) ?: [];
    }
    
    $categories = array_filter($categories, function($category) use ($id) {
        return $category['id'] !== $id;
    });
    
    file_put_contents($categoryFile, json_encode(array_values($categories), JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    echo json_encode(['success' => true, 'message' => '删除成功']);
}

// 删除楼层配置
function handleDeleteFloorConfig() {
    if (!isset($_SESSION['admin'])) {
        echo json_encode(['success' => false, 'message' => '请先登录']);
        return;
    }
    
    $id = $_POST['id'] ?? '';
    
    if (empty($id)) {
        echo json_encode(['success' => false, 'message' => 'ID不能为空']);
        return;
    }
    
    $floorFile = DATA_DIR . '/floor_config.json';
    $floors = [];
    
    if (file_exists($floorFile)) {
        $floors = json_decode(file_get_contents($floorFile), true) ?: [];
    }
    
    $floors = array_filter($floors, function($floor) use ($id) {
        return $floor['id'] !== $id;
    });
    
    file_put_contents($floorFile, json_encode(array_values($floors), JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    echo json_encode(['success' => true, 'message' => '删除成功']);
}

// 编辑商户
function handleEditMerchant() {
    if (!isset($_SESSION['admin'])) {
        echo json_encode(['success' => false, 'message' => '请先登录']);
        return;
    }
    
    $id = $_POST['id'] ?? '';
    $name = trim($_POST['name'] ?? '');
    $category = trim($_POST['category'] ?? '');
    $floor = trim($_POST['floor'] ?? '');
    
    if (empty($id) || empty($name)) {
        echo json_encode(['success' => false, 'message' => 'ID和名称不能为空']);
        return;
    }
    
    $merchantFile = DATA_DIR . '/merchants.json';
    $merchants = [];
    
    if (file_exists($merchantFile)) {
        $merchants = json_decode(file_get_contents($merchantFile), true) ?: [];
    }
    
    $found = false;
    foreach ($merchants as &$merchant) {
        if ($merchant['id'] === $id) {
            $merchant['name'] = $name;
            // 更新分类和楼层信息
            if (!empty($category)) {
                $merchant['category'] = $category;
            } else {
                unset($merchant['category']);
            }
            if (!empty($floor)) {
                $merchant['floor'] = $floor;
            } else {
                unset($merchant['floor']);
            }
            $found = true;
            break;
        }
    }
    
    if (!$found) {
        echo json_encode(['success' => false, 'message' => '商户不存在']);
        return;
    }
    
    file_put_contents($merchantFile, json_encode($merchants, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    echo json_encode(['success' => true, 'message' => '编辑成功']);
}

// 编辑分类
function handleEditCategory() {
    if (!isset($_SESSION['admin'])) {
        echo json_encode(['success' => false, 'message' => '请先登录']);
        return;
    }
    
    $id = $_POST['id'] ?? '';
    $name = trim($_POST['name'] ?? '');
    
    if (empty($id) || empty($name)) {
        echo json_encode(['success' => false, 'message' => 'ID和名称不能为空']);
        return;
    }
    
    $categoryFile = DATA_DIR . '/categories.json';
    $categories = [];
    
    if (file_exists($categoryFile)) {
        $categories = json_decode(file_get_contents($categoryFile), true) ?: [];
    }
    
    $found = false;
    foreach ($categories as &$category) {
        if ($category['id'] === $id) {
            $category['name'] = $name;
            $found = true;
            break;
        }
    }
    
    if (!$found) {
        echo json_encode(['success' => false, 'message' => '分类不存在']);
        return;
    }
    
    file_put_contents($categoryFile, json_encode($categories, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    echo json_encode(['success' => true, 'message' => '编辑成功']);
}

// 编辑楼层配置
function handleEditFloorConfig() {
    if (!isset($_SESSION['admin'])) {
        echo json_encode(['success' => false, 'message' => '请先登录']);
        return;
    }
    
    $id = $_POST['id'] ?? '';
    $name = trim($_POST['name'] ?? '');
    
    if (empty($id) || empty($name)) {
        echo json_encode(['success' => false, 'message' => 'ID和名称不能为空']);
        return;
    }
    
    $floorFile = DATA_DIR . '/floor_config.json';
    $floors = [];
    
    if (file_exists($floorFile)) {
        $floors = json_decode(file_get_contents($floorFile), true) ?: [];
    }
    
    $found = false;
    foreach ($floors as &$floor) {
        if ($floor['id'] === $id) {
            $floor['name'] = $name;
            $found = true;
            break;
        }
    }
    
    if (!$found) {
        echo json_encode(['success' => false, 'message' => '楼层配置不存在']);
        return;
    }
    
    file_put_contents($floorFile, json_encode($floors, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    echo json_encode(['success' => true, 'message' => '编辑成功']);
}

// 获取子账号列表
function handleGetSubAccounts() {
    if (!isset($_SESSION['admin'])) {
        echo json_encode(['success' => false, 'message' => '请先登录']);
        return;
    }
    
    $subAccountFile = DATA_DIR . '/sub_accounts.json';
    $subAccounts = [];
    
    if (file_exists($subAccountFile)) {
        $subAccounts = json_decode(file_get_contents($subAccountFile), true) ?: [];
    }
    
    echo json_encode(['success' => true, 'data' => $subAccounts]);
}

// 添加子账号
function handleAddSubAccount() {
    if (!isset($_SESSION['admin'])) {
        echo json_encode(['success' => false, 'message' => '请先登录']);
        return;
    }
    
    $username = trim($_POST['username'] ?? '');
    $password = trim($_POST['password'] ?? '');
    $merchantId = $_POST['merchant_id'] ?? '';
    $categoryId = $_POST['category_id'] ?? '';
    $floorId = $_POST['floor_id'] ?? '';
    $status = $_POST['status'] ?? 'active';
    
    if (empty($username) || empty($password) || empty($merchantId)) {
        echo json_encode(['success' => false, 'message' => '用户名、密码和关联商户不能为空']);
        return;
    }
    
    $subAccountFile = DATA_DIR . '/sub_accounts.json';
    $subAccounts = [];
    
    if (file_exists($subAccountFile)) {
        $subAccounts = json_decode(file_get_contents($subAccountFile), true) ?: [];
    }
    
    // 检查用户名是否已存在
    foreach ($subAccounts as $account) {
        if ($account['username'] === $username) {
            echo json_encode(['success' => false, 'message' => '用户名已存在']);
            return;
        }
    }
    
    // 获取商户信息
    $merchantFile = DATA_DIR . '/merchants.json';
    $merchants = [];
    if (file_exists($merchantFile)) {
        $merchants = json_decode(file_get_contents($merchantFile), true) ?: [];
    }
    
    $merchant = null;
    foreach ($merchants as $m) {
        if ($m['id'] === $merchantId) {
            $merchant = $m;
            break;
        }
    }
    
    if (!$merchant) {
        echo json_encode(['success' => false, 'message' => '商户不存在']);
        return;
    }
    
    // 获取分类和楼层名称
    $categoryName = '';
    $floorName = '';
    
    if ($categoryId) {
        $categoryFile = DATA_DIR . '/categories.json';
        if (file_exists($categoryFile)) {
            $categories = json_decode(file_get_contents($categoryFile), true) ?: [];
            foreach ($categories as $category) {
                if ($category['id'] === $categoryId) {
                    $categoryName = $category['name'];
                    break;
                }
            }
        }
    }
    
    if ($floorId) {
        $floorFile = DATA_DIR . '/floor_config.json';
        if (file_exists($floorFile)) {
            $floors = json_decode(file_get_contents($floorFile), true) ?: [];
            foreach ($floors as $floor) {
                if ($floor['id'] === $floorId) {
                    $floorName = $floor['name'];
                    break;
                }
            }
        }
    }
    
    $newAccount = [
        'id' => uniqid(),
        'username' => $username,
        'password' => password_hash($password, PASSWORD_DEFAULT),
        'merchant_id' => $merchantId,
        'merchant_name' => $merchant['name'],
        'category_id' => $categoryId,
        'category_name' => $categoryName,
        'floor_id' => $floorId,
        'floor_name' => $floorName,
        'status' => $status,
        'created_at' => time()
    ];
    
    $subAccounts[] = $newAccount;
    
    file_put_contents($subAccountFile, json_encode($subAccounts, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    echo json_encode(['success' => true, 'message' => '添加成功']);
}

// 编辑子账号
function handleEditSubAccount() {
    if (!isset($_SESSION['admin'])) {
        echo json_encode(['success' => false, 'message' => '请先登录']);
        return;
    }
    
    $id = $_POST['id'] ?? '';
    $username = trim($_POST['username'] ?? '');
    $password = trim($_POST['password'] ?? '');
    $merchantId = $_POST['merchant_id'] ?? '';
    $categoryId = $_POST['category_id'] ?? '';
    $floorId = $_POST['floor_id'] ?? '';
    $status = $_POST['status'] ?? 'active';
    
    if (empty($id) || empty($username) || empty($merchantId)) {
        echo json_encode(['success' => false, 'message' => 'ID、用户名和关联商户不能为空']);
        return;
    }
    
    $subAccountFile = DATA_DIR . '/sub_accounts.json';
    $subAccounts = [];
    
    if (file_exists($subAccountFile)) {
        $subAccounts = json_decode(file_get_contents($subAccountFile), true) ?: [];
    }
    
    // 检查用户名是否已被其他账号使用
    foreach ($subAccounts as $account) {
        if ($account['username'] === $username && $account['id'] !== $id) {
            echo json_encode(['success' => false, 'message' => '用户名已存在']);
            return;
        }
    }
    
    // 获取商户信息
    $merchantFile = DATA_DIR . '/merchants.json';
    $merchants = [];
    if (file_exists($merchantFile)) {
        $merchants = json_decode(file_get_contents($merchantFile), true) ?: [];
    }
    
    $merchant = null;
    foreach ($merchants as $m) {
        if ($m['id'] === $merchantId) {
            $merchant = $m;
            break;
        }
    }
    
    if (!$merchant) {
        echo json_encode(['success' => false, 'message' => '商户不存在']);
        return;
    }
    
    // 获取分类和楼层名称
    $categoryName = '';
    $floorName = '';
    
    if ($categoryId) {
        $categoryFile = DATA_DIR . '/categories.json';
        if (file_exists($categoryFile)) {
            $categories = json_decode(file_get_contents($categoryFile), true) ?: [];
            foreach ($categories as $category) {
                if ($category['id'] === $categoryId) {
                    $categoryName = $category['name'];
                    break;
                }
            }
        }
    }
    
    if ($floorId) {
        $floorFile = DATA_DIR . '/floor_config.json';
        if (file_exists($floorFile)) {
            $floors = json_decode(file_get_contents($floorFile), true) ?: [];
            foreach ($floors as $floor) {
                if ($floor['id'] === $floorId) {
                    $floorName = $floor['name'];
                    break;
                }
            }
        }
    }
    
    $found = false;
    foreach ($subAccounts as &$account) {
        if ($account['id'] === $id) {
            $account['username'] = $username;
            if (!empty($password)) {
                $account['password'] = password_hash($password, PASSWORD_DEFAULT);
            }
            $account['merchant_id'] = $merchantId;
            $account['merchant_name'] = $merchant['name'];
            $account['category_id'] = $categoryId;
            $account['category_name'] = $categoryName;
            $account['floor_id'] = $floorId;
            $account['floor_name'] = $floorName;
            $account['status'] = $status;
            $found = true;
            break;
        }
    }
    
    if (!$found) {
        echo json_encode(['success' => false, 'message' => '子账号不存在']);
        return;
    }
    
    file_put_contents($subAccountFile, json_encode($subAccounts, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    echo json_encode(['success' => true, 'message' => '更新成功']);
}

// 编辑子账号
function editSubAccount($id, $username, $password, $merchant_id, $category_id, $floor_id, $status) {
    $subAccountsFile = DATA_DIR . '/sub_accounts.json';
    
    if (!file_exists($subAccountsFile)) {
        return ['success' => false, 'message' => '子账号文件不存在'];
    }
    
    $subAccounts = json_decode(file_get_contents($subAccountsFile), true);
    if (!$subAccounts) {
        $subAccounts = [];
    }
    
    // 查找要编辑的子账号
    $accountIndex = -1;
    foreach ($subAccounts as $index => $account) {
        if ($account['id'] === $id) {
            $accountIndex = $index;
            break;
        }
    }
    
    if ($accountIndex === -1) {
        return ['success' => false, 'message' => '子账号不存在'];
    }
    
    // 检查用户名是否已存在（排除当前账号）
    foreach ($subAccounts as $index => $account) {
        if ($index !== $accountIndex && $account['username'] === $username) {
            return ['success' => false, 'message' => '用户名已存在'];
        }
    }
    
    // 获取商户信息
    $merchantInfo = null;
    if ($merchant_id) {
        $merchantFile = DATA_DIR . '/merchants.json';
        if (file_exists($merchantFile)) {
            $merchants = json_decode(file_get_contents($merchantFile), true) ?: [];
            foreach ($merchants as $merchant) {
                if ($merchant['id'] === $merchant_id) {
                    $merchantInfo = $merchant;
                    break;
                }
            }
        }
    }
    
    // 更新子账号信息
    $subAccounts[$accountIndex]['username'] = $username;
    if (!empty($password)) {
        $subAccounts[$accountIndex]['password'] = password_hash($password, PASSWORD_DEFAULT);
    }
    $subAccounts[$accountIndex]['merchant_id'] = $merchant_id;
    $subAccounts[$accountIndex]['category_id'] = $category_id;
    $subAccounts[$accountIndex]['floor_id'] = $floor_id;
    $subAccounts[$accountIndex]['status'] = $status;
    
    if ($merchantInfo) {
        $subAccounts[$accountIndex]['merchant_name'] = $merchantInfo['name'];
        // 获取分类和楼层名称
        if ($category_id) {
            $categoryFile = DATA_DIR . '/categories.json';
            if (file_exists($categoryFile)) {
                $categories = json_decode(file_get_contents($categoryFile), true) ?: [];
                foreach ($categories as $category) {
                    if ($category['id'] === $category_id) {
                        $subAccounts[$accountIndex]['category_name'] = $category['name'];
                        break;
                    }
                }
            }
        }
        
        if ($floor_id) {
            $floorFile = DATA_DIR . '/floor_config.json';
            if (file_exists($floorFile)) {
                $floors = json_decode(file_get_contents($floorFile), true) ?: [];
                foreach ($floors as $floor) {
                    if ($floor['id'] === $floor_id) {
                        $subAccounts[$accountIndex]['floor_name'] = $floor['name'];
                        break;
                    }
                }
            }
        }
    }
    
    // 保存到文件
    if (file_put_contents($subAccountsFile, json_encode($subAccounts, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE))) {
        return ['success' => true, 'message' => '子账号更新成功'];
    } else {
        return ['success' => false, 'message' => '保存失败'];
    }
}

// 删除子账号
function handleDeleteSubAccount() {
    if (!isset($_SESSION['admin'])) {
        echo json_encode(['success' => false, 'message' => '请先登录']);
        return;
    }
    
    $id = $_POST['id'] ?? '';
    
    if (empty($id)) {
        echo json_encode(['success' => false, 'message' => 'ID不能为空']);
        return;
    }
    
    $subAccountFile = DATA_DIR . '/sub_accounts.json';
    $subAccounts = [];
    
    if (file_exists($subAccountFile)) {
        $subAccounts = json_decode(file_get_contents($subAccountFile), true) ?: [];
    }
    
    $found = false;
    $subAccounts = array_filter($subAccounts, function($account) use ($id, &$found) {
        if ($account['id'] === $id) {
            $found = true;
            return false;
        }
        return true;
    });
    
    if (!$found) {
        echo json_encode(['success' => false, 'message' => '子账号不存在']);
        return;
    }
    
    // 重新索引数组
    $subAccounts = array_values($subAccounts);
    
    file_put_contents($subAccountFile, json_encode($subAccounts, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    echo json_encode(['success' => true, 'message' => '删除成功']);
}

// 获取管理员列表
function handleGetAdmins() {
    if (!isset($_SESSION['admin'])) {
        echo json_encode(['success' => false, 'message' => '未登录']);
        return;
    }
    
    $admins = json_decode(file_get_contents(ADMIN_FILE), true) ?: [];
    $adminList = [];
    
    foreach ($admins as $username => $data) {
        $adminList[] = [
            'username' => $username,
            'created_at' => $data['created_at'] ?? '',
            'status' => $data['status'] ?? 'active'
        ];
    }
    
    echo json_encode(['success' => true, 'admins' => $adminList]);
}

// 删除管理员
function handleDeleteAdmin() {
    if (!isset($_SESSION['admin'])) {
        echo json_encode(['success' => false, 'message' => '未登录']);
        return;
    }
    
    $username = $_POST['username'] ?? '';
    
    if (empty($username)) {
        echo json_encode(['success' => false, 'message' => '用户名不能为空']);
        return;
    }
    
    // 不能删除自己
    if ($username === $_SESSION['admin']) {
        echo json_encode(['success' => false, 'message' => '不能删除当前登录的管理员']);
        return;
    }
    
    $admins = json_decode(file_get_contents(ADMIN_FILE), true) ?: [];
    
    if (!isset($admins[$username])) {
        echo json_encode(['success' => false, 'message' => '管理员不存在']);
        return;
    }
    
    unset($admins[$username]);
    file_put_contents(ADMIN_FILE, json_encode($admins, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    
    echo json_encode(['success' => true, 'message' => '删除成功']);
}

// 禁用/启用管理员
function handleToggleAdmin() {
    if (!isset($_SESSION['admin'])) {
        echo json_encode(['success' => false, 'message' => '未登录']);
        return;
    }
    
    $username = $_POST['username'] ?? '';
    $action = $_POST['toggle_action'] ?? '';
    
    if (empty($username) || empty($action)) {
        echo json_encode(['success' => false, 'message' => '参数不完整']);
        return;
    }
    
    // 不能禁用自己
    if ($username === $_SESSION['admin']) {
        echo json_encode(['success' => false, 'message' => '不能禁用当前登录的管理员']);
        return;
    }
    
    $admins = json_decode(file_get_contents(ADMIN_FILE), true) ?: [];
    
    if (!isset($admins[$username])) {
        echo json_encode(['success' => false, 'message' => '管理员不存在']);
        return;
    }
    
    $admins[$username]['status'] = ($action === 'disable') ? 'disabled' : 'active';
    file_put_contents(ADMIN_FILE, json_encode($admins, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    
    $message = ($action === 'disable') ? '禁用成功' : '启用成功';
    echo json_encode(['success' => true, 'message' => $message]);
}

?>