// 全局变量
let currentFloor = 'all';
let isLoggedIn = false;
let currentAccountType = 'main'; // 'main' 或 'sub'
let merchantsList = [];
let currentAdmin = ''; // 当前登录的管理员用户名
let isSearchActive = false; // 跟踪是否处于搜索状态
let currentSearchParams = {}; // 保存当前搜索参数

// 账号排序函数
function sortAccounts(accounts) {
    return accounts.sort((a, b) => {
        // 首先按状态排序：冻结账号排在后面
        if (a.status === 'frozen' && b.status !== 'frozen') {
            return 1;
        }
        if (a.status !== 'frozen' && b.status === 'frozen') {
            return -1;
        }
        
        // 如果有排序字段，按指定字段排序
        if (currentSortField) {
            let aValue = a[currentSortField] || '';
            let bValue = b[currentSortField] || '';
            
            // 楼层排序特殊处理
            if (currentSortField === 'floor') {
                // 提取楼层中的数字和字母部分进行排序
                const parseFloor = (floor) => {
                    if (!floor) return { letter: 'Z', number: 999 };
                    
                    // 匹配 L1, L2, B1, B2 等格式
                    const match = floor.match(/^([A-Z]+)(\d+)$/);
                    if (match) {
                        return {
                            letter: match[1],
                            number: parseInt(match[2])
                        };
                    }
                    
                    // 如果不匹配标准格式，按字符串排序
                    return { letter: floor, number: 0 };
                };
                
                const aFloor = parseFloor(aValue);
                const bFloor = parseFloor(bValue);
                
                // 创建楼层排序权重：升序 B2→B1→L1→L2，降序 L2→L1→B1→B2
                const getFloorWeight = (floor) => {
                    if (floor.letter === 'B') {
                        // B楼层：B2=1, B1=2
                        return 100 - floor.number;
                    } else if (floor.letter === 'L') {
                        // L楼层：L1=101, L2=102
                        return 100 + floor.number;
                    } else {
                        // 其他楼层按字母顺序
                        return 200 + floor.letter.charCodeAt(0);
                    }
                };
                
                const aWeight = getFloorWeight(aFloor);
                const bWeight = getFloorWeight(bFloor);
                
                const comparison = aWeight - bWeight;
                return currentSortOrder === 'asc' ? comparison : -comparison;
            }
            
            // 商户名称排序 - 先按商户名称分组，再按账号排序
            if (currentSortField === 'merchant_name') {
                const merchantComparison = aValue.toString().localeCompare(bValue.toString(), 'zh-CN');
                if (merchantComparison !== 0) {
                    return currentSortOrder === 'asc' ? merchantComparison : -merchantComparison;
                }
                // 同一商户内按账号排序 - 提取数字部分进行数值比较
                const aAccount = a.account || '';
                const bAccount = b.account || '';
                const aMatch = aAccount.match(/(\d+)$/);
                const bMatch = bAccount.match(/(\d+)$/);
                if (aMatch && bMatch) {
                    const aNum = parseInt(aMatch[1]);
                    const bNum = parseInt(bMatch[1]);
                    return aNum - bNum;
                }
                return aAccount.localeCompare(bAccount);
            }
            
            // 账号排序 - 按数字部分排序
            if (currentSortField === 'account') {
                const aMatch = aValue.match(/(\d+)$/);
                const bMatch = bValue.match(/(\d+)$/);
                if (aMatch && bMatch) {
                    const aNum = parseInt(aMatch[1]);
                    const bNum = parseInt(bMatch[1]);
                    const comparison = aNum - bNum;
                    return currentSortOrder === 'asc' ? comparison : -comparison;
                }
                // 如果没有数字，按字符串排序
                const comparison = aValue.toString().localeCompare(bValue.toString());
                return currentSortOrder === 'asc' ? comparison : -comparison;
            }
            
            // 其他字段的排序
            const comparison = aValue.toString().localeCompare(bValue.toString(), 'zh-CN', { numeric: true });
            return currentSortOrder === 'asc' ? comparison : -comparison;
        }
        
        // 默认排序：先按商户名称分组，再按账号排序
        const merchantComparison = (a.merchant_name || '').localeCompare(b.merchant_name || '', 'zh-CN');
        if (merchantComparison !== 0) {
            return merchantComparison;
        }
        
        // 同一商户内按账号排序 - 提取数字部分进行数值比较
        const aAccount = a.account || '';
        const bAccount = b.account || '';
        
        // 提取账号中的数字部分
        const aMatch = aAccount.match(/(\d+)$/);
        const bMatch = bAccount.match(/(\d+)$/);
        
        if (aMatch && bMatch) {
            const aNum = parseInt(aMatch[1]);
            const bNum = parseInt(bMatch[1]);
            return aNum - bNum;
        }
        
        // 如果没有数字，按字符串排序
        return aAccount.localeCompare(bAccount);
    });
}

// 无感刷新辅助函数
function updateUIAfterLogin() {
    // 关闭登录模态框
    closeModal('loginModal');
    
    // 隐藏登录表单，显示管理界面
    const loginForm = document.getElementById('loginForm');
    const mainContent = document.getElementById('mainContent');
    
    if (loginForm) {
        loginForm.style.display = 'none';
    }
    if (mainContent) {
        mainContent.style.display = 'block';
    }
    
    // 重新加载数据
    loadFloors();
    loadAccounts();
}

function updateUIAfterLogout() {
    // 显示登录表单，隐藏管理界面
    const loginForm = document.getElementById('loginForm');
    const mainContent = document.getElementById('mainContent');
    const username = document.getElementById('username');
    const password = document.getElementById('password');
    
    if (loginForm) {
        loginForm.style.display = 'block';
    }
    if (mainContent) {
        mainContent.style.display = 'none';
    }
    
    // 清空表单数据
    if (username) {
        username.value = '';
    }
    if (password) {
        password.value = '';
    }
}

// 页面加载完成后初始化
document.addEventListener('DOMContentLoaded', function() {
    checkLoginStatus();
    loadFloors();
    loadAccounts();
    loadFloorsForFilter(); // 加载楼层筛选下拉框
    loadMerchantsForSearch(); // 加载商户数据用于搜索联想
    initEventListeners();
    
    // 绑定子账号表单提交事件
    const subAccountForm = document.getElementById('subAccountForm');
    if (subAccountForm) {
        subAccountForm.addEventListener('submit', handleSubAccountAdd);
    }
    
    // 绑定编辑子账号表单提交事件
    const editSubAccountForm = document.getElementById('editSubAccountForm');
    if (editSubAccountForm) {
        editSubAccountForm.addEventListener('submit', handleEditSubAccountSubmit);
    }
});

// 检查登录状态
function checkLoginStatus() {
    fetch('?action=check_login', {
        cache: 'no-cache',
        headers: {
            'Cache-Control': 'no-cache'
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success && data.logged_in) {
            isLoggedIn = true;
            currentAdmin = data.username || ''; // 设置当前用户名
            updateUIAfterLogin();
        } else {
            isLoggedIn = false;
            currentAdmin = '';
            updateUIAfterLogout();
        }
    })
    .catch(error => {
        // 如果请求失败，通过检查页面中是否有退出按钮来判断登录状态
        const logoutBtn = document.querySelector('button[onclick="logout()"]');
        isLoggedIn = logoutBtn !== null;
        if (isLoggedIn) {
            updateUIAfterLogin();
        } else {
            updateUIAfterLogout();
        }
    });
}

// 初始化事件监听器
function initEventListeners() {
    // 表单提交事件
    const loginForm = document.getElementById('loginForm');
    if (loginForm) {
        loginForm.addEventListener('submit', handleLogin);
    }
    
    const addForm = document.getElementById('addForm');
    if (addForm) {
        addForm.addEventListener('submit', handleAdd);
    }
    
    const changePasswordForm = document.getElementById('changePasswordForm');
    if (changePasswordForm) {
        changePasswordForm.addEventListener('submit', handleChangePassword);
    }
    
    const addAdminForm = document.getElementById('addAdminForm');
    if (addAdminForm) {
        addAdminForm.addEventListener('submit', handleAddAdmin);
    }
    
    // 绑定子账号表单提交事件
    const subAccountForm = document.getElementById('subAccountForm');
    if (subAccountForm) {
        subAccountForm.addEventListener('submit', handleSubAccountAdd);
    }
    
    // 楼层筛选变化事件
    const floorFilter = document.getElementById('floorFilter');
    if (floorFilter) {
        floorFilter.addEventListener('change', handleFloorFilterChange);
    }
    
    // 商户搜索输入事件
    const merchantSearchInput = document.getElementById('merchantSearchInput');
    if (merchantSearchInput) {
        merchantSearchInput.addEventListener('input', onMerchantSearchInput);
        merchantSearchInput.addEventListener('focus', onMerchantSearchInput);
        merchantSearchInput.addEventListener('blur', function() {
            setTimeout(hideMerchantSearchSuggestions, 200);
        });
    }
    
    // 模态框点击外部关闭
    document.addEventListener('click', function(e) {
        if (e.target.classList.contains('modal')) {
            e.target.style.display = 'none';
        }
    });
}

// 加载楼层数据 - 移除楼层标签功能
function loadFloors() {
    // 不再渲染楼层标签，直接加载所有账号数据
    loadAccounts();
}

// 加载账号数据 - 简化为加载所有数据
function loadAccounts() {
    const url = `?action=get_accounts&floor=all&_t=${Date.now()}`;
    
    fetch(url, {
        cache: 'no-cache',
        headers: {
            'Cache-Control': 'no-cache'
        }
    })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // 对账号列表进行排序
                const sortedAccounts = sortAccounts(data.accounts);
                renderAccountTable(sortedAccounts);
            } else {
                showMessage('加载数据失败', 'error');
            }
        })
        .catch(error => {
            showMessage('加载数据失败', 'error');
        });
}







// 渲染账号表格
function renderAccountTable(accounts) {
    const tableBody = document.getElementById('tableBody');
    if (!tableBody) return;
    
    if (accounts.length === 0) {
        tableBody.innerHTML = `
            <tr>
                <td colspan="${isLoggedIn ? 10 : 9}" class="empty-data">
                    <img src="assets/empty-data.svg" alt="暂无数据" class="empty-icon">
                    <div>暂无数据</div>
                </td>
            </tr>
        `;
        return;
    }
    
    let html = '';
    accounts.forEach((account, index) => {
        let statusClass, statusText;
        
        // 优先根据数据库中的status字段判断状态
        if (account.status === 'frozen') {
            statusClass = 'status-frozen';
            statusText = '已冻结';
        } else if (account.status === 'active' || !account.status) {
            // 对于active状态或没有status字段，根据创建时间判断是新增还是激活
            const createdDate = new Date(account.created_at);
            const currentDate = new Date();
            const daysDiff = Math.floor((currentDate - createdDate) / (1000 * 60 * 60 * 24));
            
            if (daysDiff <= 5) {
                statusClass = 'status-new';
                statusText = '新增';
            } else {
                statusClass = 'status-activated';
                statusText = '激活';
            }
        } else {
            // 默认状态
            statusClass = 'status-new';
            statusText = '新增';
        }
        
        html += `
            <tr>
                <td>${index + 1}</td>
                <td>${escapeHtml(account.merchant_name)}</td>
                <td>${escapeHtml(account.merchant_category_name || '-')}</td>
                <td>${escapeHtml(account.account)}</td>
                <td>${escapeHtml(account.phone)}</td>
                <td>${escapeHtml(account.floor_name || '-')}</td>
                <td>${escapeHtml(account.remark || '-')}</td>
                <td class="status-column"><span class="${statusClass}">${statusText}</span></td>
                <td>${account.updated_at}</td>
        `;
        
        if (isLoggedIn) {
            html += `
                <td class="actions-cell">
                    ${account.status === 'frozen' ? 
                        `<span class="status-text status-frozen">不可操作</span>` : 
                        `<button class="btn btn-primary" onclick="editAccount('${account.id}')">编辑</button>
                         <button class="btn btn-danger" onclick="deleteAccount('${account.id}')">冻结</button>`
                    }
                </td>
            `;
        }
        
        html += '</tr>';
    });
    
    tableBody.innerHTML = html;
}

// 搜索功能
function search() {
    const keyword = document.getElementById('searchInput').value.trim();
    
    if (!keyword) {
        showMessage('请输入搜索关键词', 'error');
        return;
    }
    
    const url = `?action=search&keyword=${encodeURIComponent(keyword)}`;
    
    fetch(url)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                renderAccountTable(data.accounts);
                // 重置楼层选择
                document.querySelectorAll('.floor-tab').forEach(tab => {
                    tab.classList.remove('active');
                });
            } else {
                showMessage(data.message || '搜索失败', 'error');
            }
        })
        .catch(error => {
            showMessage('搜索失败', 'error');
        });
}

// 显示确认对话框
function showConfirmDialog(options) {
    const { title, message, onConfirm, onCancel } = options;
    
    if (confirm(`${title}\n\n${message}`)) {
        if (onConfirm) onConfirm();
    } else {
        if (onCancel) onCancel();
    }
}

// 转义HTML字符
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// 加载管理员列表
function loadAdminList() {
    fetch('?action=get_admins')
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                renderAdminList(data.admins);
            } else {
                showMessage(data.message || '加载管理员列表失败', 'error');
            }
        })
        .catch(error => {
            console.error('加载管理员列表失败:', error);
            showMessage('加载管理员列表失败', 'error');
        });
}

// 渲染管理员列表
function renderAdminList(admins) {
    const adminList = document.getElementById('adminList');
    if (!adminList) return;
    
    if (admins.length === 0) {
        adminList.innerHTML = '<p style="text-align: center; color: #666; padding: 20px;">暂无管理员</p>';
        return;
    }
    
    adminList.innerHTML = admins.map(admin => {
        const isCurrentUser = admin.username === getCurrentUsername();
        const status = admin.status || 'active';
        const statusText = status === 'active' ? '正常' : '禁用';
        const statusClass = status === 'active' ? 'active' : 'disabled';
        
        return `
            <div class="admin-item">
                <div class="admin-info">
                    <div class="admin-username">${escapeHtml(admin.username)}</div>
                    <div class="admin-meta">
                        <span class="admin-status ${statusClass}">${statusText}</span>
                        创建时间: ${admin.created_at || '未知'}
                    </div>
                </div>
                <div class="admin-actions">
                    ${!isCurrentUser ? `
                        ${status === 'active' ? 
                            `<button class="btn btn-warning" onclick="toggleAdmin('${admin.username}', 'disable')">禁用</button>` :
                            `<button class="btn btn-success" onclick="toggleAdmin('${admin.username}', 'enable')">启用</button>`
                        }
                        <button class="btn btn-danger" onclick="deleteAdmin('${admin.username}')">删除</button>
                    ` : '<span style="color: #666; font-size: 12px;">当前用户</span>'}
                </div>
            </div>
        `;
    }).join('');
}

// 获取当前用户名（从session或其他方式）
function getCurrentUsername() {
    return currentAdmin || '';
}

// 删除管理员
function deleteAdmin(username) {
    showConfirmDialog({
        title: '确认删除',
        message: `确定要删除管理员 "${username}" 吗？此操作不可恢复。`,
        onConfirm: () => {
            fetch('?action=delete_admin', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: `username=${encodeURIComponent(username)}`
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showMessage(data.message, 'success');
                    loadAdminList(); // 重新加载列表
                } else {
                    showMessage(data.message || '删除失败', 'error');
                }
            })
            .catch(error => {
                console.error('删除管理员失败:', error);
                showMessage('删除失败', 'error');
            });
        }
    });
}

// 禁用/启用管理员
function toggleAdmin(username, action) {
    const actionText = action === 'disable' ? '禁用' : '启用';
    
    showConfirmDialog({
        title: `确认${actionText}`,
        message: `确定要${actionText}管理员 "${username}" 吗？`,
        onConfirm: () => {
            fetch('?action=toggle_admin', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: `username=${encodeURIComponent(username)}&toggle_action=${action}`
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showMessage(data.message, 'success');
                    loadAdminList(); // 重新加载列表
                } else {
                    showMessage(data.message || `${actionText}失败`, 'error');
                }
            })
            .catch(error => {
                console.error(`${actionText}管理员失败:`, error);
                showMessage(`${actionText}失败`, 'error');
            });
        }
    });
}

// 编辑账号
function editAccount(id) {
    if (!isLoggedIn) {
        showMessage('请先登录', 'error');
        return;
    }
    
    // 获取账号数据
    fetch(`?action=get_accounts&floor=all&_t=${Date.now()}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const account = data.accounts.find(acc => acc.id === id);
                if (account) {
                    // 填充编辑表单
                    document.getElementById('merchant_name').value = account.merchant_name;
                    document.getElementById('account').value = account.account;
                    document.getElementById('phone').value = account.phone;
                    document.getElementById('remark').value = account.remark || '';
                    
                    // 设置隐藏字段用于编辑
                    let editIdInput = document.getElementById('edit_id');
                    if (!editIdInput) {
                        editIdInput = document.createElement('input');
                        editIdInput.type = 'hidden';
                        editIdInput.id = 'edit_id';
                        editIdInput.name = 'edit_id';
                        document.getElementById('addForm').appendChild(editIdInput);
                    }
                    editIdInput.value = id;
                    
                    // 加载下拉框数据并设置选中值
                    loadMerchantsForSelect();
                    loadCategoriesForSelect();
                    loadFloorsForSelect();
                    
                    // 延迟设置选中值，等待下拉框加载完成
                    setTimeout(() => {
                        const categorySelect = document.getElementById('merchant_category');
                        if (categorySelect && account.merchant_category) {
                            categorySelect.value = account.merchant_category;
                        }
                        
                        const floorSelect = document.getElementById('floor');
                        if (floorSelect && account.floor) {
                            floorSelect.value = account.floor;
                        }
                    }, 500);
                    
                    // 修改模态框标题
                    const modalTitle = document.querySelector('#addModal h2');
                    if (modalTitle) {
                        modalTitle.textContent = '编辑账号';
                    }
                    
                    // 显示模态框
                    document.getElementById('addModal').style.display = 'block';
                } else {
                    showMessage('未找到账号信息', 'error');
                }
            } else {
                showMessage('获取账号信息失败', 'error');
            }
        })
        .catch(error => {
            showMessage('获取账号信息失败', 'error');
        });
}

// 清空搜索
function clearSearch() {
    // 清除商户搜索输入框
    const merchantSearchInput = document.getElementById('merchantSearchInput');
    if (merchantSearchInput) {
        merchantSearchInput.value = '';
    }
    
    // 清除楼层筛选，重置为"全部楼层"
    const floorFilter = document.getElementById('floorFilter');
    if (floorFilter) {
        floorFilter.value = '';
    }
    
    currentFloor = 'all';
    
    // 重置搜索状态
    isSearchActive = false;
    currentSearchParams = {};
    
    // 重置楼层标签
    const floorTabs = document.querySelectorAll('.floor-tab');
    floorTabs.forEach(tab => {
        tab.classList.remove('active');
    });
    
    const firstFloorTab = document.querySelector('.floor-tab');
    if (firstFloorTab) {
        firstFloorTab.classList.add('active');
    }
    
    loadAccounts();
}

// 显示登录模态框
function showLogin() {
    document.getElementById('loginModal').style.display = 'block';
}

// 处理登录
function handleLogin(e) {
    e.preventDefault();
    
    const username = document.getElementById('username').value;
    const password = document.getElementById('password').value;
    
    const formData = new FormData();
    formData.append('username', username);
    formData.append('password', password);
    
    fetch('?action=login', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showMessage('登录成功', 'success');
            // 登录成功后重新加载页面以显示正确的UI状态
            setTimeout(() => {
                window.location.reload();
            }, 1000);
        } else {
            showMessage(data.message || '登录失败', 'error');
        }
    })
    .catch(error => {
        showMessage('登录失败', 'error');
    });
}

// 退出登录
function logout() {
    fetch('?action=logout', {
        method: 'POST'
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showMessage('已退出登录', 'success');
            // 退出成功后重新加载页面以显示正确的UI状态
            setTimeout(() => {
                window.location.reload();
            }, 1000);
        } else {
            showMessage('退出失败', 'error');
        }
    })
    .catch(error => {
        showMessage('退出失败', 'error');
    });
}

// 显示新增表单
function showAddForm() {
    if (!isLoggedIn) {
        showMessage('请先登录', 'error');
        return;
    }
    
    if (currentAccountType === 'sub') {
        showSubAccountForm();
    } else {
        // 加载下拉框数据
        loadMerchantsForSelect();
        loadCategoriesForSelect();
        loadFloorsForSelect();
        
        document.getElementById('addModal').style.display = 'block';
    }
}

// 处理新增账号
function handleAdd(e) {
    e.preventDefault();
    
    const formData = new FormData();
    formData.append('merchant_name', document.getElementById('merchant_name').value);
    formData.append('merchant_category', document.getElementById('merchant_category').value);
    formData.append('account', document.getElementById('account').value);
    formData.append('phone', document.getElementById('phone').value);
    formData.append('floor', document.getElementById('floor').value);
    formData.append('remark', document.getElementById('remark').value);
    
    // 检查是否为编辑模式
    const editIdInput = document.getElementById('edit_id');
    if (editIdInput && editIdInput.value) {
        formData.append('edit_id', editIdInput.value);
    }
    
    fetch('?action=add', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showMessage(data.message || '操作成功', 'success');
            closeModal('addModal');
            document.getElementById('addForm').reset();
            
            // 清除编辑ID和重置标题
            if (editIdInput) {
                editIdInput.remove();
            }
            const modalTitle = document.querySelector('#addModal h2');
            if (modalTitle) {
                modalTitle.textContent = '新增账号';
            }
            
            // 如果当前处于搜索状态，保持搜索结果
            if (isSearchActive) {
                performSearch();
            } else {
                loadAccounts(); // 只重新加载账号数据，不再调用loadFloors
            }
        } else {
            showMessage(data.message || '操作失败', 'error');
        }
    })
    .catch(error => {
        showMessage('操作失败', 'error');
    });
}

// 现代化确认对话框
function showConfirmDialog(options) {
    const {
        title = '确认操作',
        message = '您确定要执行此操作吗？',
        confirmText = '确认',
        cancelText = '取消',
        type = 'danger', // 'danger', 'warning'
        onConfirm = () => {},
        onCancel = () => {}
    } = options;

    // 创建对话框HTML
    const dialogHTML = `
        <div id="confirmDialog" class="confirm-dialog">
            <div class="confirm-dialog-content">
                <div class="confirm-dialog-icon ${type === 'warning' ? 'warning' : ''}">
                    ${type === 'warning' ? '⚠️' : '🗑️'}
                </div>
                <div class="confirm-dialog-title">${title}</div>
                <div class="confirm-dialog-message">${message}</div>
                <div class="confirm-dialog-buttons">
                    <button class="confirm-dialog-btn cancel" onclick="hideConfirmDialog()">${cancelText}</button>
                    <button class="confirm-dialog-btn ${type === 'warning' ? 'warning' : 'confirm'}" onclick="confirmDialogAction()">${confirmText}</button>
                </div>
            </div>
        </div>
    `;

    // 移除已存在的对话框
    const existingDialog = document.getElementById('confirmDialog');
    if (existingDialog) {
        existingDialog.remove();
    }

    // 添加到页面
    document.body.insertAdjacentHTML('beforeend', dialogHTML);
    
    // 显示对话框
    const dialog = document.getElementById('confirmDialog');
    dialog.style.display = 'block';

    // 存储回调函数
    window.currentConfirmCallback = onConfirm;
    window.currentCancelCallback = onCancel;

    // 点击背景关闭
    dialog.addEventListener('click', (e) => {
        if (e.target === dialog) {
            hideConfirmDialog();
        }
    });

    // ESC键关闭
    const handleEsc = (e) => {
        if (e.key === 'Escape') {
            hideConfirmDialog();
            document.removeEventListener('keydown', handleEsc);
        }
    };
    document.addEventListener('keydown', handleEsc);
}

function hideConfirmDialog() {
    const dialog = document.getElementById('confirmDialog');
    if (dialog) {
        dialog.style.display = 'none';
        dialog.remove();
    }
    if (window.currentCancelCallback) {
        window.currentCancelCallback();
    }
    // 清理回调函数
    window.currentConfirmCallback = null;
    window.currentCancelCallback = null;
}

function confirmDialogAction() {
    if (window.currentConfirmCallback) {
        window.currentConfirmCallback();
    }
    hideConfirmDialog();
}

// 删除（冻结）账号
function deleteAccount(id) {
    showConfirmDialog({
        title: '冻结账号',
        message: '确定要冻结这个账号吗？冻结后账号将无法使用，但可以恢复。',
        confirmText: '冻结',
        cancelText: '取消',
        type: 'warning',
        onConfirm: () => {
            const formData = new FormData();
            formData.append('id', id);
            
            fetch('?action=delete', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showMessage('账号已冻结', 'success');
                    // 如果当前处于搜索状态，保持搜索结果
                    if (isSearchActive) {
                        performSearch();
                    } else {
                        loadAccounts();
                    }
                } else {
                    showMessage(data.message || '操作失败', 'error');
                }
            })
            .catch(error => {
                showMessage('操作失败', 'error');
            });
        }
    });
}

// 永久删除账号
function permanentDeleteAccount(id) {
    showConfirmDialog({
        title: '永久删除账号',
        message: '确定要永久删除这个已冻结的账号吗？此操作不可恢复！',
        confirmText: '永久删除',
        cancelText: '取消',
        type: 'danger',
        onConfirm: () => {
            const formData = new FormData();
            formData.append('id', id);
            
            fetch('?action=permanent_delete', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showMessage('账号已永久删除', 'success');
                    loadAccounts();
                } else {
                    showMessage(data.message || '删除失败', 'error');
                }
            })
            .catch(error => {
                showMessage('删除失败', 'error');
            });
        }
    });
}

// 导出数据
function exportData() {
    window.location.href = '?action=export';
}

function importData() {
    // 创建文件输入元素
    const input = document.createElement('input');
    input.type = 'file';
    input.accept = '.xlsx,.xls,.csv';
    input.onchange = function(e) {
        const file = e.target.files[0];
        if (file) {
            const formData = new FormData();
            formData.append('file', file);
            formData.append('action', 'import');
            
            fetch('', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showMessage('导入成功', 'success');
                    loadAccounts(); // 重新加载数据
                } else {
                    showMessage(data.message || '导入失败', 'error');
                }
            })
            .catch(error => {
                console.error('导入错误:', error);
                showMessage('导入失败', 'error');
            });
        }
    };
    input.click();
}

// 显示管理员面板
function showAdminPanel() {
    if (!isLoggedIn) {
        showMessage('请先登录', 'error');
        return;
    }
    document.getElementById('adminModal').style.display = 'block';
    loadAdminList(); // 加载管理员列表
}

// 处理修改密码
function handleChangePassword(e) {
    e.preventDefault();
    
    const newPassword = document.getElementById('newPassword').value;
    const confirmPassword = document.getElementById('confirmPassword').value;
    
    if (newPassword !== confirmPassword) {
        showMessage('两次输入的密码不一致', 'error');
        return;
    }
    
    const formData = new FormData();
    formData.append('new_password', newPassword);
    
    fetch('?action=change_password', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showMessage('密码修改成功', 'success');
            document.getElementById('changePasswordForm').reset();
        } else {
            showMessage(data.message || '修改失败', 'error');
        }
    })
    .catch(error => {
        showMessage('修改失败', 'error');
    });
}

// 处理添加管理员
function handleAddAdmin(e) {
    e.preventDefault();
    
    const username = document.getElementById('adminUsername').value;
    const password = document.getElementById('adminPassword').value;
    
    const formData = new FormData();
    formData.append('username', username);
    formData.append('password', password);
    
    fetch('?action=add_admin', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showMessage('管理员添加成功', 'success');
            document.getElementById('addAdminForm').reset();
        } else {
            showMessage(data.message || '添加失败', 'error');
        }
    })
    .catch(error => {
        showMessage('添加失败', 'error');
    });
}

// 关闭模态框
function closeModal(modalId) {
    document.getElementById(modalId).style.display = 'none';
}

// 显示消息
function showMessage(message, type = 'info') {
    // 移除现有消息
    const existingMessage = document.querySelector('.message');
    if (existingMessage) {
        existingMessage.remove();
    }
    
    // 创建新消息
    const messageDiv = document.createElement('div');
    messageDiv.className = `message ${type}`;
    messageDiv.textContent = message;
    
    // 插入到页面顶部
    const container = document.querySelector('.container');
    container.insertBefore(messageDiv, container.firstChild);
    
    // 3秒后自动移除
    setTimeout(() => {
        if (messageDiv.parentNode) {
            messageDiv.remove();
        }
    }, 3000);
}

// HTML转义
function escapeHtml(text) {
    if (text === null || text === undefined) {
        return '';
    }
    const map = {
        '&': '&amp;',
        '<': '&lt;',
        '>': '&gt;',
        '"': '&quot;',
        "'": '&#039;'
    };
    return String(text).replace(/[&<>"']/g, function(m) { return map[m]; });
}

// 加载商户名称到下拉框
function loadMerchantsForSelect() {
    fetch('?action=get_merchants&_t=' + Date.now(), {
        cache: 'no-cache',
        headers: {
            'Cache-Control': 'no-cache'
        }
    })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                merchantsList = data.merchants;
                const select = document.getElementById('merchant_name'); // 修复ID
                // 清空现有选项，保留默认选项
                select.innerHTML = '<option value="">请选择商户</option>';
                
                data.merchants.forEach(merchant => {
                    const option = document.createElement('option');
                    option.value = merchant.name;
                    option.textContent = merchant.name;
                    select.appendChild(option);
                });
                
                // 绑定商户选择变化事件
                select.addEventListener('change', onMerchantChange);
            }
        })
        .catch(error => {});
}

// 商户名称输入联想功能
function onMerchantInput() {
    const input = document.getElementById('merchant_name');
    const value = input.value.trim();
    const suggestionsDiv = document.getElementById('merchant_suggestions');
    
    if (value.length === 0) {
        suggestionsDiv.style.display = 'none';
        return;
    }
    
    // 过滤匹配的商户
    const filteredMerchants = merchantsList.filter(merchant => 
        merchant.name.toLowerCase().includes(value.toLowerCase())
    );
    
    // 显示建议
    if (filteredMerchants.length > 0) {
        suggestionsDiv.innerHTML = filteredMerchants.map(merchant => 
            `<div class="suggestion-item" onclick="selectMerchant('${merchant.name}')">${merchant.name}</div>`
        ).join('');
        suggestionsDiv.style.display = 'block';
    } else {
        suggestionsDiv.innerHTML = '<div class="suggestion-item not-found">未找到匹配的商户，将创建新商户</div>';
        suggestionsDiv.style.display = 'block';
    }
}

function showMerchantSuggestions() {
    const input = document.getElementById('merchant_name');
    const suggestionsDiv = document.getElementById('merchant_suggestions');
    
    if (!input || !suggestionsDiv) return;
    
    if (input.value.trim().length > 0) {
        onMerchantInput();
    } else {
        // 显示所有商户
        suggestionsDiv.innerHTML = merchantsList.map(merchant => 
            `<div class="suggestion-item" onclick="selectMerchant('${merchant.name}')">${merchant.name}</div>`
        ).join('');
        suggestionsDiv.style.display = 'block';
    }
}

function hideMerchantSuggestions() {
    setTimeout(() => {
        const suggestionsDiv = document.getElementById('merchant_suggestions');
        if (suggestionsDiv) {
            suggestionsDiv.style.display = 'none';
        }
    }, 200);
}

function selectMerchant(merchantName) {
    const input = document.getElementById('merchant_name');
    input.value = merchantName;
    
    const suggestionsDiv = document.getElementById('merchant_suggestions');
    suggestionsDiv.style.display = 'none';
    
    // 触发商户变更事件
    onMerchantChange();
}

// 商户选择变化时的处理函数
function onMerchantChange() {
    const merchantName = document.getElementById('merchant_name').value;
    if (merchantName) {
        loadCategoryForMerchant(merchantName);
        loadFloorForMerchant(merchantName);
    } else {
        // 清空分类和楼层选择
        const categorySelect = document.getElementById('merchant_category');
        const floorSelect = document.getElementById('floor');
        
        categorySelect.innerHTML = '<option value="">请先选择商户</option>';
        floorSelect.innerHTML = '<option value="">请选择楼层</option>';
        
        loadCategoriesForSelect();
        loadFloorsForSelect();
    }
}

// 根据商户加载分类
function loadCategoryForMerchant(merchantName) {
    const categorySelect = document.getElementById('merchant_category');
    
    fetch(`?action=get_merchants&_t=${Date.now()}`, {
        cache: 'no-cache',
        headers: {
            'Cache-Control': 'no-cache'
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            const merchant = data.merchants.find(m => m.name === merchantName);
            if (merchant) {
                categorySelect.innerHTML = `<option value="${merchant.category}">${merchant.category_name}</option>`;
            }
        }
    })
    .catch(error => {
        categorySelect.innerHTML = '<option value="">加载失败</option>';
    });
}

// 根据商户加载楼层
function loadFloorForMerchant(merchantName) {
    const floorSelect = document.getElementById('floor');
    
    fetch(`?action=get_merchants&_t=${Date.now()}`, {
        cache: 'no-cache',
        headers: {
            'Cache-Control': 'no-cache'
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            const merchant = data.merchants.find(m => m.name === merchantName);
            if (merchant) {
                floorSelect.innerHTML = `<option value="${merchant.floor}">${merchant.floor_name}</option>`;
            }
        }
    })
    .catch(error => {
        floorSelect.innerHTML = '<option value="">加载失败</option>';
    });
}

// 加载商户分类到下拉框
function loadCategoriesForSelect() {
    fetch('?action=get_categories&_t=' + Date.now(), {
        cache: 'no-cache',
        headers: {
            'Cache-Control': 'no-cache'
        }
    })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const select = document.getElementById('merchantCategory');
                // 清空现有选项，保留默认选项
                select.innerHTML = '<option value="">请选择商户分类</option>';
                
                data.categories.forEach(category => {
                    const option = document.createElement('option');
                    option.value = category.name;
                    option.textContent = category.name;
                    select.appendChild(option);
                });
            }
        })
        .catch(error => {});
}

// 加载楼层配置到下拉框
function loadFloorsForSelect() {
    fetch('?action=get_floor_config&_t=' + Date.now(), {
        cache: 'no-cache',
        headers: {
            'Cache-Control': 'no-cache'
        }
    })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const select = document.getElementById('floor');
                // 清空现有选项，保留默认选项
                select.innerHTML = '<option value="">请选择楼层</option>';
                
                data.floors.forEach(floor => {
                    const option = document.createElement('option');
                    option.value = floor.name;
                    option.textContent = floor.name;
                    select.appendChild(option);
                });
            }
        })
        .catch(error => {});
}

// 账号类型切换功能
function switchAccountType(type) {
    currentAccountType = type;
    
    // 更新标签样式
    document.querySelectorAll('.tab-btn').forEach(btn => {
        btn.classList.remove('active');
    });
    event.target.classList.add('active');
    
    // 切换表头
    const mainHeader = document.getElementById('mainAccountHeader');
    const subHeader = document.getElementById('subAccountHeader');
    
    if (type === 'main') {
        mainHeader.style.display = '';
        subHeader.style.display = 'none';
        // 更新按钮文本
        const addBtn = document.querySelector('button[onclick="showAddForm()"]');
        if (addBtn) addBtn.textContent = '新增账号';
    } else {
        mainHeader.style.display = 'none';
        subHeader.style.display = '';
        // 更新按钮文本
        const addBtn = document.querySelector('button[onclick="showAddForm()"]');
        if (addBtn) addBtn.textContent = '新增子账号';
    }
    
    // 重新加载数据
    if (type === 'main') {
        loadAccounts();
    } else {
        loadSubAccounts();
    }
}

// 删除子账号
function deleteSubAccount(id) {
    if (confirm('确定要删除这个子账号吗？')) {
        const formData = new FormData();
        formData.append('action', 'deleteSubAccount');
        formData.append('id', id);
        
        fetch('index.php', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showMessage('子账号删除成功', 'success');
                loadSubAccounts();
            } else {
                showMessage(data.message || '删除失败', 'error');
            }
        })
        .catch(error => {
            showMessage('删除子账号失败', 'error');
        });
    }
}
function loadSubAccounts() {
    const params = new URLSearchParams({
        action: 'getSubAccounts',
        floor: currentFloor
    });
    
    fetch(`index.php?${params}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                renderSubAccountTable(data.accounts);
            } else {
                showMessage(data.message || '加载子账号失败', 'error');
            }
        })
        .catch(error => {
            showMessage('加载子账号失败', 'error');
        });
}

// 渲染子账号表格
function renderSubAccountTable(accounts) {
    const tbody = document.getElementById('tableBody');
    
    if (accounts.length === 0) {
        tbody.innerHTML = '<tr><td colspan="7" class="empty-state">暂无子账号数据</td></tr>';
        return;
    }
    
    tbody.innerHTML = accounts.map(account => {
        let statusClass, statusText;
        
        // 根据状态显示不同的样式和文本
        switch(account.status) {
            case 'active':
                statusClass = 'status-active';
                statusText = '启用';
                break;
            case 'inactive':
                statusClass = 'status-inactive';
                statusText = '禁用';
                break;
            case 'frozen':
                statusClass = 'status-frozen';
                statusText = '已冻结';
                break;
            default:
                statusClass = 'status-active';
                statusText = '启用';
        }
        
        return `
            <tr>
                <td>${escapeHtml(account.username)}</td>
                <td>${escapeHtml(account.merchant_name || '')}</td>
                <td>${escapeHtml(account.category_name || '')}</td>
                <td>${escapeHtml(account.floor_name || '')}</td>
                <td>${account.created_at || ''}</td>
                <td><span class="${statusClass}">${statusText}</span></td>
                ${isLoggedIn ? `
                    <td class="action-buttons">
                        ${account.status === 'frozen' ? 
                            `<button class="btn btn-sm btn-warning" onclick="editSubAccount('${account.id}')">修改</button>
                             <button class="btn btn-sm btn-danger" onclick="deleteSubAccount('${account.id}')">删除</button>` : 
                            `<button class="btn btn-sm btn-warning" onclick="editSubAccount('${account.id}')">编辑</button>
                             <button class="btn btn-sm btn-danger" onclick="deleteSubAccount('${account.id}')">删除</button>`
                        }
                    </td>
                ` : ''}
            </tr>
        `;
    }).join('');
}

// 显示子账号添加表单
function showSubAccountForm() {
    loadMerchantsForSubAccount();
    document.getElementById('subAccountModal').style.display = 'block';
}

// 为子账号加载商户选项
function loadMerchantsForSubAccount() {
    fetch('index.php?action=getAccounts')
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const merchantSelect = document.getElementById('subMerchant');
                merchantSelect.innerHTML = '<option value="">请选择商户</option>';
                data.accounts.forEach(account => {
                    const option = document.createElement('option');
                    option.value = account.id;
                    option.textContent = account.merchant_name;
                    option.dataset.category = account.category_name;
                    option.dataset.categoryId = account.category_id;
                    option.dataset.floor = account.floor_name;
                    option.dataset.floorId = account.floor_id;
                    merchantSelect.appendChild(option);
                });
                
                // 监听商户选择变化
                merchantSelect.addEventListener('change', function() {
                    const selectedOption = this.options[this.selectedIndex];
                    if (selectedOption.value) {
                        document.getElementById('subCategoryDisplay').value = selectedOption.dataset.category;
                        document.getElementById('subCategoryId').value = selectedOption.dataset.categoryId;
                        document.getElementById('subFloorDisplay').value = selectedOption.dataset.floor;
                        document.getElementById('subFloorId').value = selectedOption.dataset.floorId;
                    } else {
                        document.getElementById('subCategoryDisplay').value = '';
                        document.getElementById('subCategoryId').value = '';
                        document.getElementById('subFloorDisplay').value = '';
                        document.getElementById('subFloorId').value = '';
                    }
                });
            }
        })
        .catch(error => {
        });
}

// 处理子账号添加
function handleSubAccountAdd(e) {
    e.preventDefault();
    
    const formData = new FormData();
    formData.append('action', 'addSubAccount');
    formData.append('username', document.getElementById('subUsername').value);
    formData.append('password', document.getElementById('subPassword').value);
    formData.append('merchant_id', document.getElementById('subMerchant').value);
    formData.append('status', document.getElementById('subStatus').value);
    
    fetch('index.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showMessage('子账号添加成功', 'success');
            closeModal('subAccountModal');
            document.getElementById('subAccountForm').reset();
            loadSubAccounts();
        } else {
            showMessage(data.message || '添加失败', 'error');
        }
    })
    .catch(error => {
        showMessage('添加子账号失败', 'error');
    });
}

// 编辑子账号
function editSubAccount(id) {
    // 获取子账号数据
    fetch(`index.php?action=getSubAccounts`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const account = data.data.find(acc => acc.id === id);
                if (account) {
                    // 填充编辑表单
                    document.getElementById('editSubAccountId').value = account.id;
                    document.getElementById('editSubUsername').value = account.username;
                    document.getElementById('editSubPassword').value = ''; // 密码留空
                    
                    // 加载商户选项并设置当前值
                    loadMerchantsForEditSubAccount(account);
                    
                    // 设置状态
                    document.getElementById('editSubStatus').value = account.status || 'active';
                    
                    // 显示编辑模态框
                    document.getElementById('editSubAccountModal').style.display = 'block';
                } else {
                    showMessage('子账号不存在', 'error');
                }
            } else {
                showMessage(data.message || '获取子账号信息失败', 'error');
            }
        })
        .catch(error => {
            showMessage('获取子账号信息失败', 'error');
        });
}

// 为编辑子账号加载商户选项
function loadMerchantsForEditSubAccount(account) {
    fetch('index.php?action=getAccounts')
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const merchantSelect = document.getElementById('editSubMerchant');
                merchantSelect.innerHTML = '<option value="">请选择商户</option>';
                
                data.accounts.forEach(acc => {
                    const option = document.createElement('option');
                    option.value = acc.id;
                    option.textContent = acc.merchant_name;
                    option.dataset.category = acc.category_name;
                    option.dataset.categoryId = acc.category_id;
                    option.dataset.floor = acc.floor_name;
                    option.dataset.floorId = acc.floor_id;
                    
                    // 如果是当前账号的商户，设为选中
                    if (acc.id === account.merchant_id) {
                        option.selected = true;
                    }
                    
                    merchantSelect.appendChild(option);
                });
                
                // 设置分类和楼层显示
                if (account.merchant_id) {
                    document.getElementById('editSubCategoryDisplay').value = account.category_name || '';
                    document.getElementById('editSubCategoryId').value = account.category_id || '';
                    document.getElementById('editSubFloorDisplay').value = account.floor_name || '';
                    document.getElementById('editSubFloorId').value = account.floor_id || '';
                }
                
                // 监听商户选择变化
                merchantSelect.addEventListener('change', function() {
                    const selectedOption = this.options[this.selectedIndex];
                    if (selectedOption.value) {
                        document.getElementById('editSubCategoryDisplay').value = selectedOption.dataset.category;
                        document.getElementById('editSubCategoryId').value = selectedOption.dataset.categoryId;
                        document.getElementById('editSubFloorDisplay').value = selectedOption.dataset.floor;
                        document.getElementById('editSubFloorId').value = selectedOption.dataset.floorId;
                    } else {
                        document.getElementById('editSubCategoryDisplay').value = '';
                        document.getElementById('editSubCategoryId').value = '';
                        document.getElementById('editSubFloorDisplay').value = '';
                        document.getElementById('editSubFloorId').value = '';
                    }
                });
            }
        })
        .catch(error => {
            showMessage('加载商户选项失败', 'error');
        });
}

// 处理子账号编辑表单提交
function handleEditSubAccountSubmit(e) {
    e.preventDefault();
    
    const formData = new FormData();
    formData.append('action', 'editSubAccount');
    formData.append('id', document.getElementById('editSubAccountId').value);
    formData.append('username', document.getElementById('editSubUsername').value);
    
    const password = document.getElementById('editSubPassword').value;
    if (password) {
        formData.append('password', password);
    }
    
    formData.append('merchant_id', document.getElementById('editSubMerchant').value);
    formData.append('category_id', document.getElementById('editSubCategoryId').value);
    formData.append('floor_id', document.getElementById('editSubFloorId').value);
    formData.append('status', document.getElementById('editSubStatus').value);
    
    fetch('index.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showMessage('子账号更新成功', 'success');
            closeModal('editSubAccountModal');
            loadSubAccounts();
        } else {
            showMessage(data.message || '更新失败', 'error');
        }
    })
    .catch(error => {
        showMessage('更新子账号失败', 'error');
    });
}

// 处理楼层筛选变化
function handleFloorFilterChange() {
    const floorFilter = document.getElementById('floorFilter');
    const selectedFloor = floorFilter.value;
    
    // 清空商户搜索输入框
    const merchantSearchInput = document.getElementById('merchantSearchInput');
    if (merchantSearchInput) {
        merchantSearchInput.value = '';
    }
    
    // 隐藏商户搜索建议
    hideMerchantSearchSuggestions();
    
    // 根据楼层筛选加载账号
    if (selectedFloor) {
        // 选择了特定楼层，加载该楼层的所有账号
        loadAccountsByFloor(selectedFloor);
    } else {
        // 选择了"全部楼层"，加载所有账号
        loadAccounts();
    }
}

// 根据楼层加载账号
function loadAccountsByFloor(floorId) {
    fetch(`?action=get_accounts&floor=${floorId}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                renderAccountTable(data.accounts);
            } else {
                showMessage(data.message || '加载账号失败', 'error');
            }
        })
        .catch(error => {
            console.error('加载账号失败:', error);
            showMessage('加载账号失败', 'error');
        });
}

let searchMerchantsList = [];

// 商户搜索输入处理（支持实时搜索）
let searchTimeout;
function onMerchantSearchInput() {
    const input = document.getElementById('merchantSearchInput');
    const suggestionsDiv = document.getElementById('merchantSearchSuggestions');
    
    if (!input || !suggestionsDiv) return;
    
    const inputValue = input.value.toLowerCase().trim();
    
    // 清除之前的定时器
    if (searchTimeout) {
        clearTimeout(searchTimeout);
    }
    
    if (inputValue.length === 0) {
        suggestionsDiv.style.display = 'none';
        // 如果输入为空，显示所有数据
        searchTimeout = setTimeout(() => {
            loadAccounts();
        }, 300);
        return;
    }
    
    // 过滤匹配的商户
    const filteredMerchants = searchMerchantsList.filter(merchant => 
        merchant.name.toLowerCase().includes(inputValue)
    );
    
    if (filteredMerchants.length === 0) {
        suggestionsDiv.style.display = 'none';
    } else {
        // 生成建议列表
        let suggestionsHtml = '';
        filteredMerchants.slice(0, 10).forEach(merchant => {
            suggestionsHtml += `<div class="suggestion-item" onclick="selectMerchantForSearch('${merchant.name}', '${merchant.id}')">${escapeHtml(merchant.name)}</div>`;
        });
        
        suggestionsDiv.innerHTML = suggestionsHtml;
        suggestionsDiv.style.display = 'block';
    }
    
    // 实时搜索（延迟500ms避免频繁请求）
    searchTimeout = setTimeout(() => {
        performSearch();
    }, 500);
}

function selectMerchantForSearch(merchantName, merchantId) {
    const input = document.getElementById('merchantSearchInput');
    const suggestionsDiv = document.getElementById('merchantSearchSuggestions');
    
    if (input) {
        input.value = merchantName;
        input.dataset.merchantId = merchantId;
    }
    
    if (suggestionsDiv) {
        suggestionsDiv.style.display = 'none';
    }
}

function hideMerchantSearchSuggestions() {
    const suggestionsDiv = document.getElementById('merchantSearchSuggestions');
    setTimeout(() => {
        if (suggestionsDiv) {
            suggestionsDiv.style.display = 'none';
        }
    }, 200);
}

// 为搜索加载商户数据
function loadMerchantsForSearch() {
    fetch('?action=get_merchants')
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                searchMerchantsList = data.merchants;
            }
        })
        .catch(error => {
            console.error('加载商户数据失败:', error);
        });
}

// 为搜索加载楼层数据
function loadFloorsForSearch() {
    fetch('?action=get_floors')
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const searchValue = document.getElementById('searchValue');
                searchValue.innerHTML = '<option value="">请选择楼层...</option>';
                data.floors.forEach(floor => {
                    const option = document.createElement('option');
                    option.value = floor.id;
                    option.textContent = floor.name;
                    searchValue.appendChild(option);
                });
            }
        })
        .catch(error => {
        });
}

// 执行搜索
// 加载楼层筛选下拉框
function loadFloorsForFilter() {
    fetch('?action=get_floors')
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const floorFilter = document.getElementById('floorFilter');
                if (floorFilter) {
                    floorFilter.innerHTML = '<option value="">全部楼层</option>';
                    data.floors.forEach(floor => {
                        floorFilter.innerHTML += `<option value="${floor.id}">${floor.name}</option>`;
                    });
                }
            }
        })
        .catch(error => {
            console.error('加载楼层筛选失败:', error);
        });
}

function performSearch() {
    const floorFilter = document.getElementById('floorFilter').value;
    const merchantInput = document.getElementById('merchantSearchInput');
    const searchValue = merchantInput ? merchantInput.value.trim() : '';
    
    // 保存当前搜索参数
    currentSearchParams = {
        floorFilter: floorFilter,
        searchValue: searchValue
    };
    
    // 如果没有输入商户名称且没有选择楼层，则加载所有账号
    if (!searchValue && !floorFilter) {
        isSearchActive = false;
        loadAccounts();
        return;
    }
    
    // 设置搜索状态为激活
    isSearchActive = true;
    
    // 显示搜索状态
    const tableBody = document.getElementById('tableBody');
    if (tableBody) {
        tableBody.innerHTML = '<tr><td colspan="9" style="text-align: center; padding: 20px;">搜索中...</td></tr>';
    }
    
    let url = '?action=search';
    const params = [];
    
    // 如果输入了商户名称，添加商户搜索参数
    if (searchValue) {
        params.push(`type=merchant`);
        params.push(`value=${encodeURIComponent(searchValue)}`);
    }
    
    // 如果选择了楼层，添加楼层筛选参数
    if (floorFilter) {
        params.push(`floor=${encodeURIComponent(floorFilter)}`);
    }
    
    if (params.length > 0) {
        url += '&' + params.join('&');
    }
    
    fetch(url)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // 对搜索结果进行排序
                const sortedAccounts = sortAccounts(data.accounts);
                renderAccountTable(sortedAccounts);
                // 显示搜索结果统计
                const resultCount = sortedAccounts.length;
                if (resultCount === 0) {
                    showMessage('未找到匹配的商户', 'warning');
                } else {
                    showMessage(`找到 ${resultCount} 个匹配结果`, 'success');
                }
            } else {
                showMessage(data.message || '搜索失败', 'error');
                if (tableBody) {
                    tableBody.innerHTML = '<tr><td colspan="9" style="text-align: center; padding: 20px;">搜索失败</td></tr>';
                }
            }
        })
        .catch(error => {
            console.error('搜索失败:', error);
            showMessage('搜索请求失败', 'error');
            if (tableBody) {
                tableBody.innerHTML = '<tr><td colspan="9" style="text-align: center; padding: 20px;">搜索失败</td></tr>';
            }
        });
}

// 添加排序功能
let currentSortField = '';
let currentSortOrder = 'asc';

function sortTable(field) {
    // 如果点击的是同一个字段，切换排序顺序
    if (currentSortField === field) {
        currentSortOrder = currentSortOrder === 'asc' ? 'desc' : 'asc';
    } else {
        currentSortField = field;
        currentSortOrder = 'asc';
    }
    
    // 重新加载数据并排序
    if (isSearchActive) {
        performSearch();
    } else {
        loadAccounts();
    }
}