<?php
// 不需要再次启动会话，因为主index.php已经启动了
// session_start();
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/admin-navigation.php';

if (!isAdmin()) {
    header('Location: /admin');
    exit;
}

// 处理表单提交
if ($_POST) {
    $settings = getSettings();
    
    // 处理历史版本应用
    if (isset($_POST['apply_history'])) {
        $image_type = $_POST['image_type'];
        $filename = $_POST['filename'];
        
        if ($image_type === 'background') {
            $settings['site']['current_background'] = $filename;
        } elseif ($image_type === 'avatar') {
            $settings['site']['current_avatar'] = $filename;
        }
        
        if (saveSettings($settings)) {
            echo json_encode(['success' => true, 'message' => '应用成功']);
        } else {
            echo json_encode(['success' => false, 'message' => '应用失败']);
        }
        exit;
    }
    
    if (isset($_POST['save_email'])) {
        $settings['email'] = [
            'smtp_host' => sanitizeInput($_POST['smtp_host']),
            'smtp_port' => (int)$_POST['smtp_port'],
            'smtp_secure' => sanitizeInput($_POST['smtp_secure']),
            'smtp_username' => sanitizeInput($_POST['smtp_username']),
            'smtp_password' => sanitizeInput($_POST['smtp_password']),
            'from_email' => sanitizeInput($_POST['from_email']),
            'from_name' => sanitizeInput($_POST['from_name'])
        ];
        
        if (saveSettings($settings)) {
            $success_message = '邮件设置已保存';
        } else {
            $error_message = '保存失败';
        }
    }
    
    if (isset($_POST['save_comment'])) {
        $settings['comment'] = [
            'require_email_verification' => isset($_POST['require_email_verification']),
            'auto_approve' => isset($_POST['auto_approve']),
            'verification_code_length' => (int)$_POST['verification_code_length'],
            'verification_code_expire' => (int)$_POST['verification_code_expire']
        ];
        
        if (saveSettings($settings)) {
            $success_message = '评论设置已保存';
        } else {
            $error_message = '保存失败';
        }
    }
    
    if (isset($_POST['save_site'])) {
        $settings['site'] = [
            'name' => sanitizeInput($_POST['site_name']),
            'description' => sanitizeInput($_POST['site_description']),
            'timezone' => sanitizeInput($_POST['timezone'])
        ];
        
        if (saveSettings($settings)) {
            $success_message = '站点设置已保存';
        } else {
            $error_message = '保存失败';
        }
    }
    
    // 处理图片上传
    if (isset($_POST['save_images'])) {
        // 创建src目录用于存储背景和头像
        $upload_dir = __DIR__ . '/../src/';
        if (!is_dir($upload_dir)) {
            mkdir($upload_dir, 0755, true);
        }
        
        $success = true;
        $uploaded_files = [];
        
        // 处理背景图上传
        if (isset($_FILES['background_image']) && $_FILES['background_image']['error'] === UPLOAD_ERR_OK) {
            $allowed_types = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
            $max_size = 5 * 1024 * 1024; // 5MB
            
            if (in_array($_FILES['background_image']['type'], $allowed_types) && $_FILES['background_image']['size'] <= $max_size) {
                // 使用原始文件名，添加时间戳避免冲突
                $original_name = pathinfo($_FILES['background_image']['name'], PATHINFO_FILENAME);
                $extension = pathinfo($_FILES['background_image']['name'], PATHINFO_EXTENSION);
                $filename = 'bg_' . $original_name . '_' . date('Y-m-d_H-i-s') . '.' . $extension;
                $target_path = $upload_dir . $filename;
                
                if (move_uploaded_file($_FILES['background_image']['tmp_name'], $target_path)) {
                    $uploaded_files[] = '背景图';
                    
                    // 更新当前背景图设置
                    $settings_file = __DIR__ . '/../data/settings.json';
                    $settings = json_decode(file_get_contents($settings_file), true);
                    $settings['site']['current_background'] = $filename;
                    file_put_contents($settings_file, json_encode($settings, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
                } else {
                    $success = false;
                }
            } else {
                $success = false;
                $error_message = '背景图格式不支持或文件过大';
            }
        }
        
        // 处理头像上传
        if (isset($_FILES['avatar_image']) && $_FILES['avatar_image']['error'] === UPLOAD_ERR_OK) {
            $allowed_types = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
            $max_size = 2 * 1024 * 1024; // 2MB
            
            if (in_array($_FILES['avatar_image']['type'], $allowed_types) && $_FILES['avatar_image']['size'] <= $max_size) {
                // 使用原始文件名，添加时间戳避免冲突
                $original_name = pathinfo($_FILES['avatar_image']['name'], PATHINFO_FILENAME);
                $extension = pathinfo($_FILES['avatar_image']['name'], PATHINFO_EXTENSION);
                $filename = 'avatar_' . $original_name . '_' . date('Y-m-d_H-i-s') . '.' . $extension;
                $target_path = $upload_dir . $filename;
                
                if (move_uploaded_file($_FILES['avatar_image']['tmp_name'], $target_path)) {
                    $uploaded_files[] = '头像';
                    
                    // 更新当前头像设置
                    $settings_file = __DIR__ . '/../data/settings.json';
                    $settings = json_decode(file_get_contents($settings_file), true);
                    $settings['site']['current_avatar'] = $filename;
                    file_put_contents($settings_file, json_encode($settings, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
                } else {
                    $success = false;
                }
            } else {
                $success = false;
                $error_message = '头像格式不支持或文件过大';
            }
        }
        
        if ($success && !empty($uploaded_files)) {
            $success_message = implode('、', $uploaded_files) . ' 上传成功';
        } elseif ($success) {
            $error_message = '请选择要上传的文件';
        } else {
            $error_message = $error_message ?? '上传失败';
        }
    }
}

$settings = getSettings();
?>
<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>系统设置 - <?php echo getSiteName(); ?></title>
    <link rel="stylesheet" href="/assets/css/admin.css">
</head>
<body>
    <div class="admin-container">
        <?php renderAdminNavigation('settings'); ?>
        
        <main class="admin-main">
            <div class="page-header">
                <h2>系统设置</h2>
            </div>
            
            <?php if (isset($success_message)): ?>
                <div class="alert alert-success"><?php echo $success_message; ?></div>
            <?php endif; ?>
            
            <?php if (isset($error_message)): ?>
                <div class="alert alert-error"><?php echo $error_message; ?></div>
            <?php endif; ?>
            
            <div class="admin-content system-settings">
                <div class="settings-tabs">
                <button class="tab-btn active" onclick="showTab('site')">站点设置</button>
                <button class="tab-btn" onclick="showTab('images')">图片管理</button>
                <button class="tab-btn" onclick="showTab('email')">邮件设置</button>
                <button class="tab-btn" onclick="showTab('comment')">评论设置</button>
            </div>
            
            <!-- 站点设置 -->
            <div id="site-tab" class="tab-content active">
                <form method="POST" class="settings-form">
                    <h3>站点基本信息</h3>
                    <div class="form-group">
                        <label for="site_name">站点名称</label>
                        <input type="text" id="site_name" name="site_name" value="<?php echo htmlspecialchars($settings['site']['name']); ?>" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="site_description">站点描述</label>
                        <textarea id="site_description" name="site_description" rows="3"><?php echo htmlspecialchars($settings['site']['description']); ?></textarea>
                    </div>
                    
                    <div class="form-group">
                        <label for="timezone">时区</label>
                        <select id="timezone" name="timezone">
                            <option value="Asia/Shanghai" <?php echo $settings['site']['timezone'] === 'Asia/Shanghai' ? 'selected' : ''; ?>>Asia/Shanghai</option>
                            <option value="UTC" <?php echo $settings['site']['timezone'] === 'UTC' ? 'selected' : ''; ?>>UTC</option>
                        </select>
                    </div>
                    
                    <button type="submit" name="save_site" class="btn btn-primary">保存站点设置</button>
                </form>
            </div>
            
            <!-- 图片管理 -->
            <div id="images-tab" class="tab-content">
                <form method="POST" enctype="multipart/form-data" class="settings-form">
                    <h3>站点图片管理</h3>
                    
                    <!-- 当前背景图预览 -->
                    <div class="form-group">
                        <label>当前背景图</label>
                        <div class="current-image">
                            <?php
                            $src_dir = __DIR__ . '/../src/';
                            $settings = getSettings();
                            
                            // 优先显示settings中设置的当前背景图
                            if ($settings && isset($settings['site']['current_background'])) {
                                $current_bg_file = $settings['site']['current_background'];
                                if (file_exists($src_dir . $current_bg_file)) {
                                    echo '<img src="../src/' . $current_bg_file . '" alt="当前背景图" style="max-width: 200px; max-height: 150px; object-fit: cover; border-radius: 6px;">';
                                } else {
                                    echo '<p>当前背景图文件不存在</p>';
                                }
                            } else {
                                // 回退到查找默认文件
                                $bg_files = glob($src_dir . 'backpic.*');
                                if (!empty($bg_files)) {
                                    $bg_file = basename($bg_files[0]);
                                    echo '<img src="../src/' . $bg_file . '" alt="当前背景图" style="max-width: 200px; max-height: 150px; object-fit: cover; border-radius: 6px;">';
                                } else {
                                    echo '<p>暂无背景图</p>';
                                }
                            }
                            ?>
                        </div>
                    </div>
                    
                    <!-- 背景图历史版本 -->
                    <div class="form-group">
                        <label>背景图历史版本</label>
                        <div class="image-history" id="bg-history">
                            <?php
                            $bg_history = glob($src_dir . 'bg_*.*');
                            if (!empty($bg_history)) {
                                // 按时间倒序排列
                                usort($bg_history, function($a, $b) {
                                    return filemtime($b) - filemtime($a);
                                });
                                
                                foreach ($bg_history as $bg_file) {
                                    $filename = basename($bg_file);
                                    $file_time = date('Y-m-d H:i:s', filemtime($bg_file));
                                    echo '<div class="history-item">';
                                    echo '<img src="../src/' . $filename . '" alt="历史背景图" style="width: 80px; height: 60px; object-fit: cover; border-radius: 4px;">';
                                    echo '<div class="history-info">';
                                    echo '<p>' . $file_time . '</p>';
                                    echo '<button type="button" class="btn btn-small" onclick="applyHistoryImage(\'background\', \'' . $filename . '\')">应用</button>';
                                    echo '</div>';
                                    echo '</div>';
                                }
                            } else {
                                echo '<p>暂无历史版本</p>';
                            }
                            ?>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label for="background_image">上传新背景图</label>
                        <input type="file" id="background_image" name="background_image" accept="image/*">
                        <small>支持 JPG、PNG、GIF、WebP 格式，最大 5MB</small>
                    </div>
                    
                    <!-- 当前头像预览 -->
                    <div class="form-group">
                        <label>当前头像</label>
                        <div class="current-image">
                            <?php
                            $settings = getSettings();
                            
                            // 优先显示settings中设置的当前头像
                            if ($settings && isset($settings['site']['current_avatar'])) {
                                $current_avatar_file = $settings['site']['current_avatar'];
                                if (file_exists($src_dir . $current_avatar_file)) {
                                    echo '<img src="../src/' . $current_avatar_file . '" alt="当前头像" style="width: 100px; height: 100px; object-fit: cover; border-radius: 50%;">';
                                } else {
                                    echo '<p>当前头像文件不存在</p>';
                                }
                            } else {
                                // 回退到查找默认文件
                                $avatar_files = glob($src_dir . 'avatar.*');
                                if (!empty($avatar_files)) {
                                    $avatar_file = basename($avatar_files[0]);
                                    echo '<img src="../src/' . $avatar_file . '" alt="当前头像" style="width: 100px; height: 100px; object-fit: cover; border-radius: 50%;">';
                                } else {
                                    echo '<p>暂无头像</p>';
                                }
                            }
                            ?>
                        </div>
                    </div>
                    
                    <!-- 头像历史版本 -->
                    <div class="form-group">
                        <label>头像历史版本</label>
                        <div class="image-history" id="avatar-history">
                            <?php
                            $avatar_history = glob($src_dir . 'avatar_*.*');
                            if (!empty($avatar_history)) {
                                // 按时间倒序排列
                                usort($avatar_history, function($a, $b) {
                                    return filemtime($b) - filemtime($a);
                                });
                                
                                foreach ($avatar_history as $avatar_file) {
                                    $filename = basename($avatar_file);
                                    $file_time = date('Y-m-d H:i:s', filemtime($avatar_file));
                                    echo '<div class="history-item">';
                                    echo '<img src="../src/' . $filename . '" alt="历史头像" style="width: 60px; height: 60px; object-fit: cover; border-radius: 50%;">';
                                    echo '<div class="history-info">';
                                    echo '<p>' . $file_time . '</p>';
                                    echo '<button type="button" class="btn btn-small" onclick="applyHistoryImage(\'avatar\', \'' . $filename . '\')">应用</button>';
                                    echo '</div>';
                                    echo '</div>';
                                }
                            } else {
                                echo '<p>暂无历史版本</p>';
                            }
                            ?>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label for="avatar_image">上传新头像</label>
                        <input type="file" id="avatar_image" name="avatar_image" accept="image/*">
                        <small>支持 JPG、PNG、GIF、WebP 格式，最大 2MB</small>
                    </div>
                    
                    <button type="submit" name="save_images" class="btn btn-primary">保存图片设置</button>
                </form>
            </div>
            
            <!-- 邮件设置 -->
            <div id="email-tab" class="tab-content">
                <form method="POST" class="settings-form">
                    <h3>SMTP 邮件配置</h3>
                    <div class="form-row">
                        <div class="form-group">
                            <label for="smtp_host">SMTP 服务器</label>
                            <input type="text" id="smtp_host" name="smtp_host" value="<?php echo htmlspecialchars($settings['email']['smtp_host']); ?>" placeholder="smtp.mxhichina.com">
                        </div>
                        
                        <div class="form-group">
                            <label for="smtp_port">SMTP 端口</label>
                            <input type="number" id="smtp_port" name="smtp_port" value="<?php echo $settings['email']['smtp_port']; ?>" placeholder="465">
                        </div>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label for="smtp_secure">加密方式</label>
                            <select id="smtp_secure" name="smtp_secure">
                                <option value="ssl" <?php echo $settings['email']['smtp_secure'] === 'ssl' ? 'selected' : ''; ?>>SSL</option>
                                <option value="tls" <?php echo $settings['email']['smtp_secure'] === 'tls' ? 'selected' : ''; ?>>TLS</option>
                                <option value="" <?php echo $settings['email']['smtp_secure'] === '' ? 'selected' : ''; ?>>无</option>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label for="smtp_username">SMTP 用户名</label>
                            <input type="text" id="smtp_username" name="smtp_username" value="<?php echo htmlspecialchars($settings['email']['smtp_username']); ?>">
                        </div>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label for="smtp_password">SMTP 密码</label>
                            <input type="password" id="smtp_password" name="smtp_password" value="<?php echo htmlspecialchars($settings['email']['smtp_password']); ?>">
                        </div>
                        
                        <div class="form-group">
                            <label for="from_email">发件人邮箱</label>
                            <input type="email" id="from_email" name="from_email" value="<?php echo htmlspecialchars($settings['email']['from_email']); ?>">
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label for="from_name">发件人名称</label>
                        <input type="text" id="from_name" name="from_name" value="<?php echo htmlspecialchars($settings['email']['from_name']); ?>">
                    </div>
                    
                    <div class="form-actions">
                        <button type="submit" name="save_email" class="btn btn-primary">保存邮件设置</button>
                        <button type="button" id="test-email-btn" class="btn btn-secondary">测试邮件</button>
                    </div>
                </form>
            </div>
            
            <!-- 评论设置 -->
            <div id="comment-tab" class="tab-content">
                <form method="POST" class="settings-form">
                    <h3>评论管理设置</h3>
                    <div class="form-row">
                        <div class="form-group">
                            <label>
                                <input type="checkbox" name="require_email_verification" <?php echo $settings['comment']['require_email_verification'] ? 'checked' : ''; ?>>
                                需要邮箱验证
                            </label>
                        </div>
                        
                        <div class="form-group">
                            <label>
                                <input type="checkbox" name="auto_approve" <?php echo $settings['comment']['auto_approve'] ? 'checked' : ''; ?>>
                                自动审核通过
                            </label>
                        </div>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label for="verification_code_length">验证码位数</label>
                            <select id="verification_code_length" name="verification_code_length">
                                <option value="4" <?php echo $settings['comment']['verification_code_length'] == 4 ? 'selected' : ''; ?>>4位</option>
                                <option value="6" <?php echo $settings['comment']['verification_code_length'] == 6 ? 'selected' : ''; ?>>6位</option>
                                <option value="8" <?php echo $settings['comment']['verification_code_length'] == 8 ? 'selected' : ''; ?>>8位</option>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label for="verification_code_expire">验证码有效期（秒）</label>
                            <input type="number" id="verification_code_expire" name="verification_code_expire" value="<?php echo $settings['comment']['verification_code_expire']; ?>" min="60" max="3600">
                        </div>
                    </div>
                    
                    <button type="submit" name="save_comment" class="btn btn-primary">保存评论设置</button>
                </form>
            </div>
            </div>
        </main>
    </div>

    <!-- 测试邮件交互式对话框 -->
    <div id="test-email-dialog" class="modal-overlay" style="display: none;">
        <div class="modal-dialog">
            <div class="modal-header">
                <h3>📧 测试邮件</h3>
                <button type="button" class="close-btn" onclick="closeTestEmailDialog()">&times;</button>
            </div>
            <div class="modal-body">
                <p>请输入接收测试邮件的邮箱地址：</p>
                <input type="email" id="test-email-input" placeholder="example@domain.com" autocomplete="email">
                <div id="email-error" class="error-message" style="display: none;"></div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" onclick="closeTestEmailDialog()">取消</button>
                <button type="button" id="confirm-send-email" class="btn btn-primary">发送测试邮件</button>
            </div>
        </div>
    </div>

    <style>
        .modal-overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.5);
            display: flex;
            justify-content: center;
            align-items: center;
            z-index: 1000;
        }
        
        .modal-dialog {
            background: white;
            border-radius: 8px;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.3);
            width: 90%;
            max-width: 400px;
            animation: modalSlideIn 0.3s ease-out;
        }
        
        @keyframes modalSlideIn {
            from {
                opacity: 0;
                transform: translateY(-50px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        .modal-header {
            padding: 20px 20px 10px;
            border-bottom: 1px solid #eee;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .modal-header h3 {
            margin: 0;
            color: #333;
            font-size: 18px;
        }
        
        .close-btn {
            background: none;
            border: none;
            font-size: 24px;
            cursor: pointer;
            color: #999;
            padding: 0;
            width: 30px;
            height: 30px;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .close-btn:hover {
            color: #333;
        }
        
        .modal-body {
            padding: 20px;
        }
        
        .modal-body p {
            margin: 0 0 15px;
            color: #666;
        }
        
        .modal-body input {
            width: 100%;
            padding: 10px;
            border: 2px solid #ddd;
            border-radius: 4px;
            font-size: 14px;
            box-sizing: border-box;
        }
        
        .modal-body input:focus {
            outline: none;
            border-color: #007cba;
        }
        
        .error-message {
            color: #d63384;
            font-size: 12px;
            margin-top: 5px;
        }
        
        .modal-footer {
            padding: 10px 20px 20px;
            display: flex;
            justify-content: flex-end;
            gap: 10px;
        }
        
        .modal-footer .btn {
            padding: 8px 16px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-size: 14px;
        }
        
        .btn-secondary {
            background: #6c757d;
            color: white;
        }
        
        .btn-secondary:hover {
            background: #5a6268;
        }
        
        .btn-primary {
            background: #007cba;
            color: white;
        }
        
        .btn-primary:hover {
            background: #005a87;
        }
        
        .btn-primary:disabled {
            background: #ccc;
            cursor: not-allowed;
        }
    </style>

    <script>
        function showTab(tabName) {
            // 隐藏所有标签页内容
            document.querySelectorAll('.tab-content').forEach(tab => {
                tab.classList.remove('active');
            });
            
            // 移除所有按钮的活动状态
            document.querySelectorAll('.tab-btn').forEach(btn => {
                btn.classList.remove('active');
            });
            
            // 显示选中的标签页
            document.getElementById(tabName + '-tab').classList.add('active');
            event.target.classList.add('active');
        }
        
        // 测试邮件功能 - 使用自定义交互式对话框
        document.getElementById('test-email-btn').addEventListener('click', function() {
            showTestEmailDialog();
        });
        
        // 显示测试邮件对话框
        function showTestEmailDialog() {
            document.getElementById('test-email-dialog').style.display = 'flex';
            document.getElementById('test-email-input').focus();
            document.getElementById('email-error').style.display = 'none';
            document.getElementById('test-email-input').value = '';
        }
        
        // 关闭测试邮件对话框
        function closeTestEmailDialog() {
            document.getElementById('test-email-dialog').style.display = 'none';
            document.getElementById('email-error').style.display = 'none';
        }
        
        // 点击背景关闭对话框
        document.getElementById('test-email-dialog').addEventListener('click', function(e) {
            if (e.target === this) {
                closeTestEmailDialog();
            }
        });
        
        // ESC键关闭对话框
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape' && document.getElementById('test-email-dialog').style.display === 'flex') {
                closeTestEmailDialog();
            }
        });
        
        // 回车键发送邮件
        document.getElementById('test-email-input').addEventListener('keydown', function(e) {
            if (e.key === 'Enter') {
                document.getElementById('confirm-send-email').click();
            }
        });
        
        // 应用历史图片
        function applyHistoryImage(type, filename) {
            if (!confirm('确定要应用这个' + (type === 'background' ? '背景图' : '头像') + '吗？')) {
                return;
            }
            
            const formData = new FormData();
            formData.append('apply_history', '1');
            formData.append('image_type', type);
            formData.append('filename', filename);
            
            fetch('/admin/settings.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.text())
            .then(data => {
                // 刷新页面以显示更新后的图片
                location.reload();
            })
            .catch(error => {
                alert('应用失败：' + error.message);
            });
        }
        
        // 确认发送测试邮件
        document.getElementById('confirm-send-email').addEventListener('click', function() {
            const email = document.getElementById('test-email-input').value.trim();
            const errorDiv = document.getElementById('email-error');
            const sendBtn = this;
            
            // 清除之前的错误信息
            errorDiv.style.display = 'none';
            
            // 验证邮箱地址
            if (!email) {
                errorDiv.textContent = '请输入邮箱地址';
                errorDiv.style.display = 'block';
                return;
            }
            
            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            if (!emailRegex.test(email)) {
                errorDiv.textContent = '请输入有效的邮箱地址';
                errorDiv.style.display = 'block';
                return;
            }
            
            // 显示发送中状态
            const originalText = sendBtn.textContent;
            sendBtn.textContent = '发送中...';
            sendBtn.disabled = true;
            
            // 发送测试邮件请求
            fetch('/admin/test-email.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: 'test_email=' + encodeURIComponent(email)
            })
            .then(response => response.json())
            .then(data => {
                // 恢复按钮状态
                sendBtn.textContent = originalText;
                sendBtn.disabled = false;
                
                // 关闭对话框
                closeTestEmailDialog();
                
                // 显示结果
                if (data.success) {
                    alert('✅ ' + data.message);
                } else {
                    alert('❌ ' + data.message);
                }
            })
            .catch(error => {
                // 恢复按钮状态
                sendBtn.textContent = originalText;
                sendBtn.disabled = false;
                
                errorDiv.textContent = '发送失败：' + error.message;
                errorDiv.style.display = 'block';
            });
        });
    </script>
</body>
</html>