<?php
// Suppress all error output to ensure a clean JSON response
ini_set('display_errors', 0);
error_reporting(0);

header('Content-Type: application/json');

$response = ['success' => false, 'message' => 'An unknown error occurred.'];

try {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('Invalid request method.');
    }

    $data = json_decode(file_get_contents('php://input'), true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception('Invalid JSON received.');
    }

    if (!isset($data['passwordHash']) || !isset($data['redirectUrl']) || !isset($data['salt'])) {
        throw new Exception('Missing required fields: passwordHash, redirectUrl, or salt.');
    }

    $configFile = __DIR__ . '/../.config/auth/config.json';
    $configDir = dirname($configFile);

    // Check if directory exists and is writable
    if (!is_dir($configDir)) {
        // Attempt to create the directory with permissions
        if (!@mkdir($configDir, 0755, true)) {
            throw new Exception('Failed to create config directory. Check permissions.');
        }
    }

    // Check directory permissions
    if (!is_writable($configDir)) {
        throw new Exception('Config directory is not writable. Current permissions: ' . substr(sprintf('%o', fileperms($configDir)), -4));
    }

    $config = [];
    if (file_exists($configFile)) {
        // Check if file is readable
        if (!is_readable($configFile)) {
            throw new Exception('Config file is not readable. Check permissions.');
        }
        $config = json_decode(file_get_contents($configFile), true) ?? [];
        unset($config['credentials']); // Remove old structure if it exists
    }

    if (!empty($data['passwordHash'])) {
        $config['passwordHash'] = $data['passwordHash'];
    }
    $config['redirectUrl'] = $data['redirectUrl'];
    $config['salt'] = $data['salt'];

    // Check if file exists and is writable
    if (file_exists($configFile) && !is_writable($configFile)) {
        throw new Exception('Config file is not writable. Current permissions: ' . substr(sprintf('%o', fileperms($configFile)), -4));
    }

    // Attempt to write to the file
    $jsonContent = json_encode($config, JSON_PRETTY_PRINT);
    if (@file_put_contents($configFile, $jsonContent) === false) {
        $error = error_get_last();
        throw new Exception('Failed to write to config file. Error: ' . ($error['message'] ?? 'Unknown error'));
    }

    // Set proper permissions after writing
    @chmod($configFile, 0666);

    $response['success'] = true;
    $response['message'] = 'Config updated successfully';

} catch (Exception $e) {
    // If any error occurs, capture it and set it as the response message
    $response['message'] = $e->getMessage();
}

echo json_encode($response);
?>