/**
 * 前端路由保护
 * 防止未经认证直接访问hash路由
 */

(function() {
    'use strict';
    
    // 检查是否已认证
    function isAuthenticated() {
        const authToken = getCookie('auth_token');
        const userEmail = getCookie('user_email');
        return authToken && userEmail;
    }
    
    // 获取cookie值
    function getCookie(name) {
        const value = `; ${document.cookie}`;
        const parts = value.split(`; ${name}=`);
        if (parts.length === 2) return parts.pop().split(';').shift();
        return null;
    }
    
    // 检查当前URL是否包含受保护的hash路由
    function hasProtectedHashRoute() {
        const hash = window.location.hash;
        const protectedRoutes = ['#/login', '#/dashboard', '#/admin', '#/panel'];
        
        return protectedRoutes.some(route => hash.startsWith(route));
    }
    
    // 重定向到登录页面
    function redirectToLogin() {
        // 清除hash
        if (window.location.hash) {
            window.location.hash = '';
        }
        // 重定向到根路径
        window.location.href = '/';
    }
    
    // 路由保护检查
    function checkRouteAccess() {
        // 如果有受保护的hash路由但未认证，则重定向
        if (hasProtectedHashRoute() && !isAuthenticated()) {
            console.warn('未经认证访问受保护路由，重定向到登录页面');
            redirectToLogin();
            return false;
        }
        return true;
    }
    
    // 监听hash变化
    function setupHashChangeListener() {
        window.addEventListener('hashchange', function() {
            if (!checkRouteAccess()) {
                return;
            }
        });
    }
    
    // 监听页面加载
    function setupPageLoadListener() {
        document.addEventListener('DOMContentLoaded', function() {
            checkRouteAccess();
        });
        
        // 如果DOM已经加载完成
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', checkRouteAccess);
        } else {
            checkRouteAccess();
        }
    }
    
    // 防止通过JavaScript直接设置hash
    function protectHashManipulation() {
        const originalPushState = history.pushState;
        const originalReplaceState = history.replaceState;
        
        history.pushState = function(state, title, url) {
            if (url && url.includes('#/') && !isAuthenticated()) {
                console.warn('阻止未认证用户通过pushState访问受保护路由');
                redirectToLogin();
                return;
            }
            return originalPushState.apply(this, arguments);
        };
        
        history.replaceState = function(state, title, url) {
            if (url && url.includes('#/') && !isAuthenticated()) {
                console.warn('阻止未认证用户通过replaceState访问受保护路由');
                redirectToLogin();
                return;
            }
            return originalReplaceState.apply(this, arguments);
        };
    }
    
    // 初始化路由保护
    function initRouteGuard() {
        setupPageLoadListener();
        setupHashChangeListener();
        protectHashManipulation();
        
        // 立即检查当前路由
        checkRouteAccess();
    }
    
    // 启动路由保护
    initRouteGuard();
    
    // 导出给全局使用
    window.RouteGuard = {
        isAuthenticated: isAuthenticated,
        checkRouteAccess: checkRouteAccess,
        redirectToLogin: redirectToLogin
    };
    
})();