// 数据库版本安全认证系统
class DatabaseAuthSystem {
    constructor() {
        this.maxAttempts = 3;
        this.attemptCount = 0;
        this.lockoutTime = 5 * 60 * 1000; // 5分钟锁定
        this.init();
    }

    async init() {
        console.log('DatabaseAuthSystem 初始化开始');
        this.bindEvents();
        
        // 确保登录按钮默认启用
        const loginBtn = document.getElementById('loginBtn');
        if (loginBtn) {
            loginBtn.disabled = false;
            loginBtn.textContent = '登录';
            console.log('登录按钮已启用');
        } else {
            console.error('找不到登录按钮元素');
        }
        
        this.checkLockout();
        
        // 检查数据库连接状态
        await this.checkDatabaseConnection();
        console.log('DatabaseAuthSystem 初始化完成');
    }

    async checkDatabaseConnection() {
        try {
            const response = await fetch('/api/db-test', {
                method: 'GET',
                headers: {
                    'Content-Type': 'application/json'
                }
            });
            
            if (!response.ok) {
                console.warn('数据库连接测试失败，但不影响登录功能');
                return;
            }
            
            const result = await response.json();
            if (!result.connected) {
                console.warn('数据库连接失败，但登录功能仍可使用');
            }
        } catch (error) {
            console.error('数据库连接检查错误:', error);
            // 不再显示错误信息或禁用登录按钮
        }
    }

    bindEvents() {
        console.log('开始绑定事件');
        const form = document.getElementById('loginForm');
        const loginBtn = document.getElementById('loginBtn');

        // 只绑定表单提交事件，这是最可靠的方式
        if (form) {
            form.addEventListener('submit', (e) => {
                e.preventDefault();
                console.log('表单提交事件触发');
                this.handleLogin();
            });
            console.log('表单提交事件已绑定');
        } else {
            console.error('找不到表单元素');
        }

        // 确保登录按钮可用
        if (loginBtn) {
            loginBtn.disabled = false;
            loginBtn.type = 'submit'; // 确保按钮类型为submit
            console.log('登录按钮已设置为可用状态');
        } else {
            console.error('找不到登录按钮');
        }
    }

    async handleLogin() {
        console.log('handleLogin 方法被调用');
        if (this.isLockedOut()) {
            this.showError('登录尝试过多，请稍后再试。');
            return;
        }

        const email = document.getElementById('email')?.value?.trim();
        const password = document.getElementById('password')?.value;

        if (!email || !password) {
            this.showError('请输入邮箱和密码');
            return;
        }

        // 验证邮箱格式
        if (!this.isValidEmail(email)) {
            this.showError('请输入有效的邮箱地址');
            return;
        }

        this.showLoading(true);

        try {
            const result = await this.validateCredentials(email, password);
            // 如果result为undefined，说明已经显示了503页面，不需要进一步处理
            if (result === undefined) {
                return;
            }
            
            if (result.success) {
                this.onLoginSuccess(result);
            } else {
                this.onLoginFailure(result.message || '登录失败');
            }
        } catch (error) {
            console.error('登录错误:', error);
            this.onLoginFailure('网络错误，请稍后重试');
        } finally {
            this.showLoading(false);
        }
    }

    async validateCredentials(email, password) {
        try {
            const response = await fetch('/api/login', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    email: email,
                    password: password
                })
            });

            // 检查响应状态
            if (response.status === 401) {
                // 认证失败，直接显示503页面
                this.show503Page();
                return; // 不返回任何值，因为页面已经被替换
            }

            if (response.status === 503) {
                // 尝试解析JSON响应，如果失败则返回通用错误
                try {
                    const errorData = await response.json();
                    return {
                        success: false,
                        message: errorData.message || '服务暂时不可用，请稍后重试'
                    };
                } catch (e) {
                    return {
                        success: false,
                        message: '服务暂时不可用，请稍后重试'
                    };
                }
            }

            if (!response.ok) {
                throw new Error('网络请求失败');
            }

            const result = await response.json();
            
            // 检查是否需要显示503页面
            if (result && result.show_503) {
                this.show503Page();
                return; // 不返回任何值，因为页面已经被替换
            }
            
            return result;
        } catch (error) {
            console.error('验证请求错误:', error);
            throw error;
        }
    }

    isValidEmail(email) {
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return emailRegex.test(email);
    }

    onLoginSuccess(result) {
        // 清除错误信息
        this.showError('');
        
        // 显示成功信息
        this.showSuccess('登录成功！正在跳转到面板...');
        
        // 重置尝试次数
        this.attemptCount = 0;
        localStorage.removeItem('loginAttempts');
        localStorage.removeItem('lockoutTime');
        
        // 延迟跳转，确保cookie设置完成
        console.log('登录成功，准备跳转到:', result.redirect);
        
        setTimeout(() => {
            // 登录成功后跳转到dashboard页面
            window.location.href = '/dashboard';
        }, 1000); // 减少延迟时间
    }

    // 显示503页面内容
    show503Page() {
        const html503 = `<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Service Unavailable</title>
    <link rel="icon" href="src/favicon.svg" type="image/svg+xml">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <style>
        body {
            font-family: Arial, sans-serif;
            background-color: #f4f4f4;
            color: #333;
            display: flex;
            justify-content: center;
            align-items: center;
            min-height: 100vh;
            margin: 0;
            text-align: center;
        }
        .container {
            background-color: #fff;
            padding: 40px;
            border-radius: 8px;
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.1);
        }
        h1 {
            font-size: 4em;
            color: #e74c3c;
            margin-bottom: 10px;
        }
        p {
            font-size: 1.2em;
            margin-bottom: 20px;
        }
        a {
            color: #3498db;
            text-decoration: none;
        }
        a:hover {
            text-decoration: underline;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>503</h1>
        <p>拒绝了您的访问。</p>
        <p>无访问权限或鉴权失败。</p>
        <p>请联系管理员。</p>
        <!--<p><a href="/">返回首页</a></p>-->
        <p>The service is temporarily unavailable. Please try again later.</p>
    </div>
    <script>
        setTimeout(function() {
            window.location.href = 'https://www.cquilm.com';
        }, 15000);
    </script>
</body>
</html>`;
        document.open();
        document.write(html503);
        document.close();
    }

    onLoginFailure(message) {
        this.attemptCount++;
        localStorage.setItem('loginAttempts', this.attemptCount.toString());
        
        // 记录失败尝试
        this.recordFailedAttempt();
        
        this.showLoading(false);
        
        // 直接显示503页面
        this.show503Page();
    }

    recordFailedAttempt() {
        const email = document.getElementById('email')?.value?.trim() || '';
        
        // 发送失败记录到服务器
        fetch('/tools/log_failed_attempt.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                timestamp: Date.now(),
                email: email,
                type: 'database_auth'
            })
        }).catch(error => {
            console.error('记录失败尝试时出错:', error);
        });
    }

    checkLockout() {
        const lockoutTime = localStorage.getItem('lockoutTime');
        const attempts = localStorage.getItem('loginAttempts');
        const loginBtn = document.getElementById('loginBtn');
        
        if (lockoutTime && Date.now() < parseInt(lockoutTime)) {
            const remainingTime = Math.ceil((parseInt(lockoutTime) - Date.now()) / 60000);
            this.showError(`账户已锁定，请在 ${remainingTime} 分钟后重试。`);
            
            if (loginBtn) loginBtn.disabled = true;
            
            // 设置定时器
            const timeLeft = parseInt(lockoutTime) - Date.now();
            setTimeout(() => {
                if (loginBtn) loginBtn.disabled = false;
                this.showError('');
                localStorage.removeItem('lockoutTime');
                localStorage.removeItem('loginAttempts');
            }, timeLeft);
        } else {
            // 没有锁定，确保登录按钮是启用的
            if (loginBtn) loginBtn.disabled = false;
            if (attempts) {
                this.attemptCount = parseInt(attempts);
            }
        }
    }

    isLockedOut() {
        const lockoutTime = localStorage.getItem('lockoutTime');
        return lockoutTime && Date.now() < parseInt(lockoutTime);
    }

    showError(message) {
        const errorDiv = document.getElementById('errorMessage');
        if (errorDiv) {
            errorDiv.textContent = message;
            errorDiv.style.display = message ? 'block' : 'none';
        }
    }

    showSuccess(message) {
        const successDiv = document.getElementById('successMessage');
        if (successDiv) {
            successDiv.textContent = message;
            successDiv.style.display = message ? 'block' : 'none';
        }
    }

    showLoading(show) {
        const loginBtn = document.getElementById('loginBtn');
        const loadingSpinner = document.querySelector('.loading-spinner');
        
        if (loginBtn) {
            loginBtn.disabled = show;
            loginBtn.textContent = show ? '登录中...' : '登录';
        }
        
        if (loadingSpinner) {
            loadingSpinner.style.display = show ? 'inline-block' : 'none';
        }
    }

    delay(ms) {
        return new Promise(resolve => setTimeout(resolve, ms));
    }
}

// 页面加载完成后初始化
document.addEventListener('DOMContentLoaded', () => {
    console.log('DOM内容已加载');
    setTimeout(() => {
        console.log('开始初始化DatabaseAuthSystem');
        new DatabaseAuthSystem();
    }, 100); // 延迟100ms确保所有元素都已渲染
});

// 防止控制台信息泄露
(function() {
    const originalLog = console.log;
    const originalError = console.error;
    const originalWarn = console.warn;
    
    console.log = function() {};
    console.error = function() {};
    console.warn = function() {};
    
    // 在开发环境下可以取消注释以下行来启用控制台
    if (window.location.hostname === 'localhost') {
        console.log = originalLog;
        console.error = originalError;
        console.warn = originalWarn;
    }
})();

console.clear();
console.log('%c🚀 欢迎来到外太空！', 'color: #667eea; font-size: 20px; font-weight: bold;');
console.log('%c如果你看到这条消息，说明你已经偏离了正常轨道...', 'color: #764ba2; font-size: 14px;');