<?php
// 调试信息
error_log("auth_check_db.php accessed - REQUEST_URI: " . $_SERVER['REQUEST_URI']);

session_start();

// 引入数据库认证类
require_once '.config/database/db_auth.php';

// 读取认证配置
function getAuthConfig() {
    $configPath = '.config/auth/config.json';
    if (file_exists($configPath)) {
        $config = json_decode(file_get_contents($configPath), true);
        return $config;
    }
    return ['redirectUrl' => 'https://www.023021.xyz']; // 默认值
}

// 检查是否已认证
function isAuthenticated() {
    // 只检查session认证，禁用cookie自动登录
    if (isset($_SESSION['authenticated']) && $_SESSION['authenticated'] === true) {
        return true;
    }
    
    return false;
}

// 获取用户邮箱
function getUserEmail() {
    return isset($_COOKIE['user_email']) ? urldecode($_COOKIE['user_email']) : null;
}

// 决定用户应该看到什么界面
function shouldShowDashboard($email) {
    try {
        $dbAuth = new DatabaseAuth();
        $daysUntilExpiry = $dbAuth->getDaysUntilExpiry($email);
        
        // 如果无法获取到期时间，默认显示仪表板
        if ($daysUntilExpiry === null) {
            return true;
        }
        
        // 大于5天：直接显示xboard
        // 小于等于5天：显示仪表板（包括过期的情况）
        return $daysUntilExpiry <= 5;
    } catch (Exception $e) {
        // 出错时默认显示仪表板
        return true;
    }
}

// 获取用户状态信息（用于仪表板显示）
function getUserStatus($email) {
    try {
        $dbAuth = new DatabaseAuth();
        $daysUntilExpiry = $dbAuth->getDaysUntilExpiry($email);
        $expiryDate = $dbAuth->getUserExpiryDate($email);
        $userName = $dbAuth->getUserName($email);
        
        if ($daysUntilExpiry === null) {
            return [
                'status' => 'unknown',
                'days' => null,
                'expiry_date' => null,
                'user_name' => $userName,
                'background_color' => '#f8f9fa'
            ];
        }
        
        if ($daysUntilExpiry < 0) {
            // 已过期
            return [
                'status' => 'expired',
                'days' => abs($daysUntilExpiry),
                'expiry_date' => $expiryDate,
                'user_name' => $userName,
                'background_color' => '#ffebee' // 红色背景
            ];
        } elseif ($daysUntilExpiry <= 1) {
            // 1天内到期
            return [
                'status' => 'critical',
                'days' => $daysUntilExpiry,
                'expiry_date' => $expiryDate,
                'user_name' => $userName,
                'background_color' => '#fff3e0' // 黄色背景
            ];
        } elseif ($daysUntilExpiry <= 5) {
            // 2-5天到期
            return [
                'status' => 'warning',
                'days' => $daysUntilExpiry,
                'expiry_date' => $expiryDate,
                'user_name' => $userName,
                'background_color' => '#e8f5e8' // 绿色背景
            ];
        } else {
            // 大于5天，正常状态
            return [
                'status' => 'normal',
                'days' => $daysUntilExpiry,
                'expiry_date' => $expiryDate,
                'user_name' => $userName,
                'background_color' => '#f8f9fa'
            ];
        }
    } catch (Exception $e) {
        return [
            'status' => 'error',
            'days' => null,
            'expiry_date' => null,
            'user_name' => null,
            'background_color' => '#f8f9fa'
        ];
    }
}

// 处理数据库连接测试
function handleDatabaseTest() {
    try {
        $dbAuth = new DatabaseAuth();
        $testResult = $dbAuth->testConnection();
        
        $isConnected = isset($testResult['connected']) ? $testResult['connected'] : false;
        
        header('Content-Type: application/json');
        echo json_encode([
            'connected' => $isConnected,
            'message' => $isConnected ? '数据库连接正常' : '数据库连接失败',
            'details' => $testResult
        ]);
        exit;
    } catch (Exception $e) {
        header('Content-Type: application/json');
        echo json_encode([
            'connected' => false,
            'message' => '数据库连接错误: ' . $e->getMessage()
        ]);
        exit;
    }
}

// 处理登录
function handleLogin() {
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        // 支持JSON和表单数据两种格式
        $input = json_decode(file_get_contents('php://input'), true);
        
        // 如果JSON解析失败，尝试从POST数据获取
        if (!$input) {
            $input = $_POST;
        }
        
        if (isset($input['email']) && isset($input['password'])) {
            $email = trim($input['email']);
            $password = $input['password'];
            
            // 验证邮箱格式
            if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                header('Content-Type: application/json');
                echo json_encode([
                    'success' => false,
                    'message' => '邮箱格式不正确'
                ]);
                exit;
            }
            
            try {
                // 使用数据库认证
                $dbAuth = new DatabaseAuth();
                
                // 先检查数据库连接
                if (!$dbAuth->isConnected()) {
                    header('Content-Type: application/json');
                    http_response_code(503);
                    echo json_encode([
                        'success' => false,
                        'message' => '数据库连接失败，请稍后重试'
                    ]);
                    exit;
                }
                
                $userInfo = $dbAuth->validateUser($email, $password);
                error_log("Login attempt - Email: $email, Password: $password, UserInfo: " . json_encode($userInfo));
                if ($userInfo) {
                    // 登录成功
                    $_SESSION['authenticated'] = true;
                    $_SESSION['user_email'] = $email;
                    $_SESSION['user_password'] = $password; // 存储密码用于自动登录xboard
                    $_SESSION['login_time'] = time();
                    $_SESSION['auth_type'] = 'database';
                    $_SESSION['xboard_auto_login'] = true; // 标记需要自动登录xboard
                    
                    // 保存用户到期信息
                    if (isset($userInfo['expired_at'])) {
                        $_SESSION['user_expired_at'] = $userInfo['expired_at'];
                    }
                    
                    // 立即写入session数据
                    session_write_close();
                    session_start(); // 重新开始session
                    
                    // 设置会话cookie（浏览器关闭时自动清除）
                    $token = hash('sha256', 'authenticated_' . date('Y-m-d') . '_' . $email);
                    setcookie('auth_token', $token, 0, '/', '', false, true);
                    setcookie('user_email', $email, 0, '/', '', false, true);
                    
                    header('Content-Type: application/json');
                    echo json_encode([
                        'success' => true,
                        'message' => '登录成功，正在加载...'
                    ]);
                    exit;
                } else {
                    // 登录失败，返回特殊状态码让前端显示503页面
                    http_response_code(401); // 使用401状态码表示认证失败
                    header('Content-Type: application/json');
                    echo json_encode([
                        'success' => false,
                        'message' => '邮箱或密码错误',
                        'show_503' => true // 标记需要显示503页面
                    ]);
                    exit;
                }
            } catch (Exception $e) {
                // 数据库连接或查询错误
                error_log('Database auth error: ' . $e->getMessage());
                header('Content-Type: application/json');
                echo json_encode([
                    'success' => false,
                    'message' => '服务器错误，请稍后重试'
                ]);
                exit;
            }
        } else {
            // 缺少邮箱或密码
            header('Content-Type: application/json');
            echo json_encode([
                'success' => false,
                'message' => '请提供邮箱和密码'
            ]);
            exit;
        }
    } else {
        // 不支持的请求方法
        header('Content-Type: application/json');
        echo json_encode([
            'success' => false,
            'message' => '不支持的请求方法'
        ]);
        exit;
    }
}

// 处理配置管理API
function handleConfigAPI() {
    header('Content-Type: application/json');
    
    if ($_SERVER['REQUEST_METHOD'] === 'GET') {
        // 获取当前配置
        try {
            $authConfig = getAuthConfig();
            echo json_encode([
                'success' => true,
                'config' => $authConfig
            ]);
        } catch (Exception $e) {
            echo json_encode([
                'success' => false,
                'message' => '读取配置失败: ' . $e->getMessage()
            ]);
        }
    } elseif ($_SERVER['REQUEST_METHOD'] === 'POST') {
        // 保存配置
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (isset($input['redirectUrl'])) {
            $redirectUrl = trim($input['redirectUrl']);
            
            // 验证URL格式
            if (!filter_var($redirectUrl, FILTER_VALIDATE_URL)) {
                echo json_encode([
                    'success' => false,
                    'message' => '无效的网址格式'
                ]);
                exit;
            }
            
            try {
                // 读取当前配置
                $configPath = '.config/auth/config.json';
                $currentConfig = [];
                
                if (file_exists($configPath)) {
                    $currentConfig = json_decode(file_get_contents($configPath), true) ?: [];
                }
                
                // 更新跳转网址
                $currentConfig['redirectUrl'] = $redirectUrl;
                
                // 保存配置
                if (file_put_contents($configPath, json_encode($currentConfig, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES))) {
                    echo json_encode([
                        'success' => true,
                        'message' => '配置保存成功'
                    ]);
                } else {
                    echo json_encode([
                        'success' => false,
                        'message' => '配置文件写入失败'
                    ]);
                }
            } catch (Exception $e) {
                echo json_encode([
                    'success' => false,
                    'message' => '保存配置失败: ' . $e->getMessage()
                ]);
            }
        } else {
            echo json_encode([
                'success' => false,
                'message' => '缺少必要参数'
            ]);
        }
    } else {
        echo json_encode([
            'success' => false,
            'message' => '不支持的请求方法'
        ]);
    }
    exit;
}

// 处理用户到期信息API
function handleUserExpiryAPI() {
    header('Content-Type: application/json');
    
    if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
        echo json_encode([
            'success' => false,
            'message' => '不支持的请求方法'
        ]);
        exit;
    }
    
    // 检查用户是否已认证
    if (!isAuthenticated()) {
        http_response_code(401);
        echo json_encode([
            'success' => false,
            'message' => '未认证'
        ]);
        exit;
    }
    
    // 从session获取用户邮箱
    if (!isset($_SESSION['user_email'])) {
        echo json_encode([
            'success' => false,
            'message' => '用户信息不完整'
        ]);
        exit;
    }
    
    try {
        $dbAuth = new DatabaseAuth();
        $expiryInfo = $dbAuth->getUserExpiryInfo($_SESSION['user_email']);
        
        if ($expiryInfo) {
            echo json_encode([
                'success' => true,
                'expiry_info' => $expiryInfo
            ]);
        } else {
            echo json_encode([
                'success' => false,
                'message' => '无法获取到期信息'
            ]);
        }
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'message' => '服务器错误: ' . $e->getMessage()
        ]);
    }
    exit;
}

// 处理不同的请求路径
$requestUri = $_SERVER['REQUEST_URI'];
$path = parse_url($requestUri, PHP_URL_PATH);

// 对于PHP内置服务器，检查查询参数中的路径
if (isset($_GET['path'])) {
    $path = $_GET['path'];
}

// 调试信息
error_log("Parsed path: " . $path);

// 移除调试信息

// 处理静态资源文件（无需认证）
$staticExtensions = ['.js', '.css', '.png', '.jpg', '.jpeg', '.gif', '.svg', '.ico', '.woff', '.woff2', '.ttf', '.eot'];
$pathExtension = strtolower(pathinfo($path, PATHINFO_EXTENSION));

// 排除xboard资源文件和uploads文件，这些需要通过代理访问
$isXboardAsset = strpos($path, '/theme/Xboard/assets/') === 0 || 
                 strpos($path, '/proxy/theme/Xboard/assets/') === 0 ||
                 strpos($path, '/uploads/') === 0 ||
                 strpos($path, '/proxy/uploads/') === 0;

error_log("Static resource check - Extension: " . $pathExtension . ", isXboardAsset: " . ($isXboardAsset ? 'true' : 'false'));

if (in_array('.' . $pathExtension, $staticExtensions) && !$isXboardAsset) {
    $filePath = __DIR__ . $path;
    
    if (file_exists($filePath) && is_file($filePath)) {
        // 设置正确的Content-Type
        $mimeTypes = [
            'js' => 'application/javascript',
            'css' => 'text/css',
            'png' => 'image/png',
            'jpg' => 'image/jpeg',
            'jpeg' => 'image/jpeg',
            'gif' => 'image/gif',
            'svg' => 'image/svg+xml',
            'ico' => 'image/x-icon',
            'woff' => 'font/woff',
            'woff2' => 'font/woff2',
            'ttf' => 'font/ttf',
            'eot' => 'application/vnd.ms-fontobject'
        ];
        
        $contentType = isset($mimeTypes[$pathExtension]) ? $mimeTypes[$pathExtension] : 'application/octet-stream';
        header('Content-Type: ' . $contentType);
        
        // 设置缓存头
        header('Cache-Control: public, max-age=3600');
        header('Expires: ' . gmdate('D, d M Y H:i:s', time() + 3600) . ' GMT');
        
        readfile($filePath);
        exit;
    } else {
        // 文件不存在，返回404
        http_response_code(404);
        echo 'File not found';
        exit;
    }
}



// 处理数据库测试API请求
if ($path === '/api/db-test') {
    handleDatabaseTest();
}

error_log("Passed db-test check");

// 处理登录API请求
if ($path === '/api/login' || $path === '/login') {
    handleLogin();
}

error_log("Passed login check");

// 处理配置管理API请求
if ($path === '/api/config') {
    handleConfigAPI();
}

// 处理用户到期信息API请求
if ($path === '/api/user-expiry') {
    if (isAuthenticated()) {
        $userEmail = getUserEmail();
        $userStatus = getUserStatus($userEmail);
        
        header('Content-Type: application/json');
        echo json_encode([
            'success' => true,
            'expiry_info' => [
                'days_remaining' => $userStatus['days'],
                'expiry_date' => $userStatus['expiry_date'],
                'status' => $userStatus['status'],
                'user_name' => $userStatus['user_name'] ?: $userEmail,
                'show_panel_button' => true,
                'background_color' => $userStatus['background_color']
            ]
        ]);
    } else {
        http_response_code(401);
        header('Content-Type: application/json');
        echo json_encode(['success' => false, 'message' => '未授权访问']);
    }
    exit;
}

// 处理退出登录API请求
if ($path === '/api/logout') {
    // 支持GET和POST方法
    if ($_SERVER['REQUEST_METHOD'] === 'POST' || $_SERVER['REQUEST_METHOD'] === 'GET') {
        // 清理session
        session_destroy();
        
        // 清理cookies
        setcookie('auth_token', '', time() - 3600, '/', '', false, true);
        setcookie('user_email', '', time() - 3600, '/', '', false, true);
        setcookie('PHPSESSID', '', time() - 3600, '/', '', false, true);
        
        header('Content-Type: application/json');
        echo json_encode(['success' => true, 'message' => '退出登录成功']);
        exit;
    } else {
        header('Content-Type: application/json');
        http_response_code(405);
        echo json_encode(['success' => false, 'message' => '不支持的请求方法']);
        exit;
    }
}

// 处理xboard资源路径 - 需要通过代理访问
error_log("Checking first Xboard assets handler for: " . $path);
if (strpos($path, '/theme/Xboard/assets/') === 0) {
    error_log("First Xboard assets handler matched: " . $path);
    if (isAuthenticated()) {
        error_log("User authenticated, including proxy.php");
        // 已认证，通过代理访问xboard资源
        // 设置标志让proxy.php知道它是被auth_check_db.php调用的
        $_SERVER['CALLED_FROM_AUTH_CHECK'] = true;
        include 'admin/proxy.php';
        exit;
    } else {
        error_log("User not authenticated for Xboard assets");
        // 未认证，返回403
        http_response_code(403);
        echo 'Access denied';
        exit;
    }
}

// 处理xboard上传文件路径 - 需要通过代理访问
error_log("Checking uploads handler for: " . $path);
if (strpos($path, '/uploads/') === 0) {
    error_log("Uploads handler matched: " . $path);
    if (isAuthenticated()) {
        error_log("User authenticated for uploads, including proxy.php");
        // 已认证，通过代理访问xboard上传文件
        // 设置标志让proxy.php知道它是被auth_check_db.php调用的
        $_SERVER['CALLED_FROM_AUTH_CHECK'] = true;
        include 'admin/proxy.php';
        exit;
    } else {
        error_log("User not authenticated for uploads");
        // 未认证，返回403
        http_response_code(403);
        echo 'Access denied';
        exit;
    }
}

// 配置页面访问
if ($path === '/config' || $path === '/config.html') {
    include 'pages/config.html';
    exit;
}

// 处理 /proxy 路径及其子路径
if ($path === '/proxy' || strpos($path, '/proxy/') === 0) {
    if (isAuthenticated()) {
        // 已认证用户，显示xboard
        $_SERVER['CALLED_FROM_AUTH_CHECK'] = true;
        
        if ($path === '/proxy') {
            // 访问 /proxy 时，代理到目标站点的根路径
            $_SERVER['REQUEST_URI'] = '/';
        } else {
            // 访问 /proxy/xxx 时，去掉 /proxy 前缀，代理到目标站点的对应路径
            $targetPath = substr($path, 6); // 去掉 '/proxy' 前缀
            $_SERVER['REQUEST_URI'] = $targetPath;
        }
        
        include 'admin/proxy.php';
        exit;
    } else {
        // 未认证用户，重定向到首页
        header('Location: /');
        exit;
    }
}

// 处理 /proxy.php/ 路径（API重定向后的路径）
if (strpos($path, '/proxy.php/') === 0) {
    if (isAuthenticated()) {
        // 已认证用户，去掉 /proxy.php 前缀，代理到目标站点的对应路径
        $_SERVER['CALLED_FROM_AUTH_CHECK'] = true;
        $targetPath = substr($path, 10); // 去掉 '/proxy.php' 前缀
        $_SERVER['REQUEST_URI'] = $targetPath;
        
        include 'admin/proxy.php';
        exit;
    } else {
        // 未认证用户，重定向到首页
        header('Location: /');
        exit;
    }
}

// 处理 /theme/Xboard/ 路径（Xboard静态资源）
error_log("Checking if path starts with /theme/Xboard/: " . $path);
if (strpos($path, '/theme/Xboard/') === 0) {
    error_log("Processing Xboard theme resource: " . $path);
    if (isAuthenticated()) {
        error_log("User authenticated, proxying to: " . $path);
        // 已认证用户，通过代理访问Xboard资源
        $_SERVER['CALLED_FROM_AUTH_CHECK'] = true;
        $_SERVER['REQUEST_URI'] = $path; // 保持原路径
        
        include 'admin/proxy.php';
        exit;
    } else {
        error_log("User not authenticated for Xboard resource");
        // 未认证用户，返回403
        http_response_code(403);
        echo 'Access denied';
        exit;
    }
}

// 处理 /proxy 路径 - 强制跳转到鉴权页面
if ($path === '/proxy' || $path === '/proxy/') {
    // 清除session
    session_destroy();
    session_start(); // 重新开始一个新的session
    
    // 清除认证相关的cookie
    setcookie('auth_token', '', time() - 3600, '/', '', false, true);
    setcookie('user_email', '', time() - 3600, '/', '', false, true);
    
    // 强制跳转到鉴权页面
    header('Location: /');
    exit;
}

// Dashboard页面访问
if ($path === '/dashboard' || $path === '/dashboard.html') {
    if (isAuthenticated()) {
        include 'pages/dashboard.html';
        exit;
    } else {
        // 未认证，重定向到登录页面
        header('Location: /');
        exit;
    }
}

// 根路径访问处理
if ($path === '/' || $path === '/index.php') {
    if (isAuthenticated()) {
        $userEmail = getUserEmail();
        
        // 根据用户到期时间决定显示什么界面
        if (shouldShowDashboard($userEmail)) {
            // 显示仪表板（到期时间 <= 5天或已过期）
            header('Location: /dashboard');
            exit;
        } else {
            // 直接显示xboard（到期时间 > 5天）
            $_SERVER['CALLED_FROM_AUTH_CHECK'] = true;
            // 设置默认路由为dashboard
            $_SERVER['XBOARD_DEFAULT_ROUTE'] = '/dashboard';
            include 'admin/proxy.php';
            exit;
        }
    } else {
        // 未认证，检查是否试图直接访问hash路由
        $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';
        $referer = $_SERVER['HTTP_REFERER'] ?? '';
        
        // 如果URL包含hash符号或者referer包含hash路由，显示403错误
        if (strpos($_SERVER['REQUEST_URI'], '#') !== false || 
            (strpos($referer, '#/') !== false && strpos($referer, 'proxy.digac.edu.kg') !== false)) {
            
            http_response_code(403);
            echo '<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>访问被拒绝</title>
    <style>
        body { font-family: Arial, sans-serif; text-align: center; padding: 50px; background: #f5f5f5; }
        .error-container { background: white; padding: 40px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); max-width: 500px; margin: 0 auto; }
        h1 { color: #dc3545; margin-bottom: 20px; }
        p { color: #666; margin-bottom: 20px; }
        .login-link { background: #007bff; color: white; text-decoration: none; padding: 10px 20px; border-radius: 4px; display: inline-block; }
        .login-link:hover { background: #0056b3; }
    </style>
</head>
<body>
    <div class="error-container">
        <h1>🚫 访问被拒绝</h1>
        <p>您需要先进行身份验证才能访问此页面。</p>
        <p>请通过正常渠道登录系统。</p>
        <a href="/" class="login-link">返回登录页面</a>
    </div>
</body>
</html>';
            exit;
        }
        
        // 正常显示登录页面
        include 'index.html';
        exit;
    }
}

// 其他所有路径的处理
if (isAuthenticated()) {
    // 已认证，通过反代访问
    // 设置标志让proxy.php知道它是被auth_check_db.php调用的
    $_SERVER['CALLED_FROM_AUTH_CHECK'] = true;
    include 'admin/proxy.php';
    exit;
} else {
    // 未认证，显示503错误页面（拒绝访问）
    http_response_code(503);
    include '503.html';
    exit;
}

?>