<?php
session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);

/**
 * Simple proxy that forwards request headers (including Cookie),
 * preserves Set-Cookie in responses, rewrites Location headers
 * to keep redirects inside the proxy, and handles missing config/cURL.
 * Includes auto-login functionality for xboard.
 */

// 全局变量存储xboard的认证cookie
$xboardAuthCookies = '';

// 检查是否需要自动登录xboard
if (isset($_SESSION['xboard_auto_login']) && $_SESSION['xboard_auto_login'] === true) {
    // 执行自动登录
    $email = $_SESSION['user_email'] ?? '';
    $password = $_SESSION['user_password'] ?? '';
    
    if ($email && $password) {
        // 获取目标URL配置
        $configFile = __DIR__ . '/../.config/auth/config.json';
        if (file_exists($configFile)) {
            $config = json_decode(file_get_contents($configFile), true);
            $targetBase = rtrim($config['redirectUrl'] ?? '', '/');
            
            if ($targetBase) {
                // 尝试自动登录xboard
                $loginUrl = $targetBase . '/api/v1/passport/auth/login';
                
                $loginData = json_encode([
                    'email' => $email,
                    'password' => $password
                ]);
                
                $ch = curl_init();
                curl_setopt($ch, CURLOPT_URL, $loginUrl);
                curl_setopt($ch, CURLOPT_POST, true);
                curl_setopt($ch, CURLOPT_POSTFIELDS, $loginData);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_HEADER, true);
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
                curl_setopt($ch, CURLOPT_HTTPHEADER, [
                    'Content-Type: application/json',
                    'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36'
                ]);
                
                $response = curl_exec($ch);
                $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                $headerSize = curl_getinfo($ch, CURLINFO_HEADER_SIZE);
                curl_close($ch);
                
                if ($response && $httpCode == 200) {
                    // 分离响应头和响应体
                    $headers = substr($response, 0, $headerSize);
                    $body = substr($response, $headerSize);
                    
                    // 检查登录是否成功
                    $responseData = json_decode($body, true);
                    if (isset($responseData['data']) && isset($responseData['data']['token'])) {
                        // 登录成功，提取Cookie用于后续请求
                        preg_match_all('/Set-Cookie:\s*([^;\r\n]+)/i', $headers, $matches);
                        if (!empty($matches[1])) {
                            $xboardAuthCookies = implode('; ', $matches[1]);
                            // 同时设置给用户浏览器
                            foreach ($matches[1] as $cookie) {
                                $cookieParts = explode('=', $cookie, 2);
                                if (count($cookieParts) == 2) {
                                    setcookie($cookieParts[0], $cookieParts[1], time() + (24 * 60 * 60), '/', '', false, true);
                                }
                            }
                        }
                        
                        // 清除自动登录标记
                        $_SESSION['xboard_auto_login'] = false;
                        
                        // 自动登录成功后，继续处理当前请求而不是重定向
                        // 这样可以避免前端路由问题
                    }
                }
            }
        }
    }
}

$configFile = __DIR__ . '/../.config/auth/config.json';
if (!file_exists($configFile)) {
    http_response_code(500);
    die('Config file not found.');
}
$config = json_decode(file_get_contents($configFile), true);
$targetBase = rtrim($config['redirectUrl'] ?? '', '/');
if (!$targetBase) {
    http_response_code(500);
    die('Missing redirectUrl in config.');
}

// Build target URL by appending the request path
$requestUri = $_SERVER['REQUEST_URI'] ?? '/';
$path = parse_url($requestUri, PHP_URL_PATH);

// 对于PHP内置服务器，检查查询参数中的路径
if (isset($_GET['path'])) {
    $path = $_GET['path'];
}

// 如果是已认证用户访问根路径，且没有自动登录标记，直接代理到dashboard
if (($path === '/' || $path === '') && (!isset($_SESSION['xboard_auto_login']) || $_SESSION['xboard_auto_login'] !== true)) {
    $path = '/#/dashboard';
}

// 如果是访问/panel路径，重定向到xboard dashboard
if ($path === '/panel') {
    $path = '/#/dashboard';
}

// Special handling for favicon.ico - redirect to local favicon.svg
if ($path === '/favicon.ico' || $path === 'favicon.ico') {
    header('Location: /src/favicon.svg');
    exit;
}

// Special handling for local files (config, auth system, tools, etc.)
$localFilePaths = ['/.config/', '.config/', '/auth.js', 'auth.js', '/tools/', 'tools/', '/src/', 'src/'];
foreach ($localFilePaths as $localPrefix) {
    if (strpos($path, $localPrefix) === 0) {
        $localPath = __DIR__ . $path;
        if (file_exists($localPath)) {
            $extension = pathinfo($localPath, PATHINFO_EXTENSION);
            $mimeType = 'text/plain';
            switch ($extension) {
                case 'json':
                    $mimeType = 'application/json';
                    break;
                case 'js':
                    $mimeType = 'application/javascript';
                    break;
                case 'html':
                    $mimeType = 'text/html';
                    break;
                case 'css':
                    $mimeType = 'text/css';
                    break;
                case 'svg':
                    $mimeType = 'image/svg+xml';
                    break;
                case 'png':
                    $mimeType = 'image/png';
                    break;
                case 'jpg':
                case 'jpeg':
                    $mimeType = 'image/jpeg';
                    break;
            }
            header('Content-Type: ' . $mimeType);
            readfile($localPath);
            exit;
        } else {
            http_response_code(404);
            die('Local file not found.');
        }
    }
}

$targetUrl = $targetBase . ($path === '' ? '/' : $path);
error_log("Proxy: Target URL: $targetUrl");
error_log("Proxy: Request method: " . ($_SERVER['REQUEST_METHOD'] ?? 'GET'));

// Method and body
$method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
$requestBody = '';
if ($method === 'POST' || $method === 'PUT' || $method === 'PATCH') {
    $requestBody = file_get_contents('php://input');
}

// Forward request headers (preserve Cookie and other important headers)
$requestHeaders = [];
$hasCookie = false;
if (function_exists('getallheaders')) {
    foreach (getallheaders() as $name => $value) {
        $lname = strtolower($name);
        // Skip headers that should be regenerated by proxy
        if (in_array($lname, ['host', 'content-length', 'connection', 'accept-encoding'])) {
            continue;
        }
        
        // 如果有xboard认证cookie，合并到现有cookie中
        if ($lname === 'cookie' && !empty($xboardAuthCookies)) {
            $value = $xboardAuthCookies . '; ' . $value;
            $hasCookie = true;
        }
        
        $requestHeaders[] = $name . ': ' . $value;
    }
}

// 如果没有cookie header但有xboard认证cookie，添加它
if (!$hasCookie && !empty($xboardAuthCookies)) {
    $requestHeaders[] = 'Cookie: ' . $xboardAuthCookies;
}

// Add User-Agent if not present
$hasUserAgent = false;
foreach ($requestHeaders as $header) {
    if (stripos($header, 'user-agent:') === 0) {
        $hasUserAgent = true;
        break;
    }
}
if (!$hasUserAgent) {
    $requestHeaders[] = 'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36';
}

// Prepare proxy URL variables for later use
$scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https://' : 'http://';
$proxyHost = $_SERVER['HTTP_HOST'] ?? 'localhost';

// 当通过auth_check_db.php调用时，我们不需要添加脚本路径前缀
// 因为所有路径都会通过auth_check_db.php进行路由
$isCalledFromAuthCheck = isset($_SERVER['CALLED_FROM_AUTH_CHECK']) && $_SERVER['CALLED_FROM_AUTH_CHECK'];
if ($isCalledFromAuthCheck) {
    $proxyScript = ''; // 不添加前缀
    $proxyUrl = $scheme . $proxyHost;
} else {
    $proxyScript = $_SERVER['SCRIPT_NAME'] ?? '/proxy.php'; // e.g., /proxy.php
    // Ensure proxyScript starts with a slash
    if (substr($proxyScript, 0, 1) !== '/') {
        $proxyScript = '/' . $proxyScript;
    }
    $proxyUrl = $scheme . $proxyHost . $proxyScript;
}



// Ensure Host header matches target host
$targetHost = parse_url($targetBase, PHP_URL_HOST);
if ($targetHost) {
    $requestHeaders[] = 'Host: ' . $targetHost;
}

// Handle referer rewriting
if (isset($_SERVER['HTTP_REFERER'])) {
    $referer = $_SERVER['HTTP_REFERER'];
    // Rewrite referer to point to target domain
    $referer = str_ireplace($scheme . $proxyHost . $proxyScript, $targetBase, $referer);
    $requestHeaders[] = 'Referer: ' . $referer;
}

// Create stream context for file_get_contents
$contextOptions = [
    'http' => [
        'method' => $method,
        'header' => implode("\r\n", $requestHeaders),
        'timeout' => 30,
        'ignore_errors' => true,
    ],
    'ssl' => [
        'verify_peer' => false,
        'verify_peer_name' => false,
    ]
];

if ($requestBody !== '' && $method !== 'HEAD') {
    $contextOptions['http']['content'] = $requestBody;
}

$context = stream_context_create($contextOptions);

// Execute the request
error_log("Proxy: Making request to $targetUrl using file_get_contents");
$body = file_get_contents($targetUrl, false, $context);
$responseHeaders = $http_response_header ?? [];

error_log("Proxy: Response received, body length: " . strlen($body ?: ''));

// Check for errors
if ($body === false) {
    error_log("Proxy: file_get_contents failed for URL: $targetUrl");
    http_response_code(502);
    die("Failed to fetch content from target URL: $targetUrl");
}

// Parse status code from response headers
$status = 200;
if (!empty($responseHeaders)) {
    $statusLine = $responseHeaders[0];
    if (preg_match('/HTTP\/\d\.\d\s+(\d+)/', $statusLine, $matches)) {
        $status = (int)$matches[1];
    }
}

error_log("Proxy: HTTP status: $status");

// Convert response headers to string format for compatibility
$headerText = implode("\r\n", $responseHeaders) . "\r\n\r\n";

// Proxy URL variables are already defined above

// Send response status
http_response_code($status);
error_log("Proxy: Setting HTTP status code to: $status");

// Parse and forward headers, and find Content-Type
$contentType = '';
$lines = preg_split("/\r\n|\n|\r/", $headerText);
foreach ($lines as $line) {
    if (trim($line) === '') continue;
    if (stripos($line, 'HTTP/') === 0) continue;

    // skip hop-by-hop headers that should be regenerated by the proxy
    if (stripos($line, 'Transfer-Encoding:') === 0 ||
        stripos($line, 'Content-Length:') === 0 ||
        stripos($line, 'Content-Encoding:') === 0 ||
        stripos($line, 'Connection:') === 0) {
        continue;
    }

    // Extract Content-Type for body rewriting later
    if (stripos($line, 'Content-Type:') === 0) {
        $contentType = trim(substr($line, strlen('Content-Type:')));
    }
    
    // Rewrite Location to stay behind proxy
    if (stripos($line, 'Location:') === 0) {
        $location = trim(substr($line, strlen('Location:')));
        // Replace target base URL with proxy script URL
        $rewritten = str_ireplace($targetBase, $proxyUrl, $location);
        header('Location: ' . $rewritten, true);
        continue;
    }

    header($line, false);
}

// Rewrite URLs in various content types
$needsRewriting = false;
if (stripos($contentType, 'text/html') !== false || 
    stripos($contentType, 'text/css') !== false || 
    stripos($contentType, 'application/javascript') !== false ||
    stripos($contentType, 'text/javascript') !== false) {
    $needsRewriting = true;
}

if ($needsRewriting) {
    // First, handle relative paths in JavaScript and CSS before absolute URL replacement
    
    // For JavaScript content, handle common URL patterns
    // Temporarily disabled to avoid conflicts with absolute URL replacement
    // if (stripos($contentType, 'javascript') !== false) {
    //     $body = preg_replace('#(?<=[\s,\[\{])(["\'])(/[^/])#', '$1' . $proxyScript . '$2', $body);
    // }
    
    // For CSS content, handle url() references
    if (stripos($contentType, 'text/css') !== false && !$isCalledFromAuthCheck) {
        // Handle url("/path") and url('/path') in CSS
        $body = preg_replace('#url\((["\']?)/#i', 'url($1' . $proxyScript . '/', $body);
    }
    
    // For HTML content, handle attributes
    if (stripos($contentType, 'text/html') !== false && stripos($contentType, 'javascript') === false) {
        // 只有在不是通过auth_check_db.php调用时才重写相对路径
        if (!$isCalledFromAuthCheck) {
            // Prepend proxy script path to root-relative URLs in common attributes
            // Only match when it's actually in an HTML tag (preceded by < and tag name)
            $body = preg_replace('#(<[^>]*\s)(href|src|action|formaction|background|data-src|data-url)=(["\'])/#i', '$1$2=$3' . $proxyScript . '/', $body);
            
            // Handle meta refresh redirects
            // Only match when it's actually in a meta tag
            $body = preg_replace('#(<meta[^>]*content=(["\'])\d+;\s*url=)/#i', '$1' . $proxyScript . '/', $body);
        }
    }
    
    // Finally, replace absolute URLs pointing to the target with our proxy URL
    $targetHost = parse_url($targetBase, PHP_URL_HOST);
    if ($targetHost) {
        // Only replace URLs that exactly match our target base URL
        $body = str_ireplace($targetBase, $proxyUrl, $body);
        // Also handle protocol-relative URLs for the target host only
        $body = str_ireplace('//' . $targetHost, $proxyUrl, $body);
    }
    
    // 只有在不是通过auth_check_db.php调用时才进行这些URL重写
    if (!$isCalledFromAuthCheck) {
        // Handle URLs pointing to the current proxy domain, but avoid double-rewriting
        // Only rewrite if the URL doesn't already contain the proxy script path
        $currentHost = $_SERVER['HTTP_HOST'] ?? '';
        if ($currentHost) {
            // Only rewrite URLs that don't already contain our proxy script path
            $proxyScriptPath = parse_url($proxyUrl, PHP_URL_PATH);
            
            // More precise patterns to avoid rewriting already correct proxy URLs
            // We need to ensure we don't match URLs that already contain the full proxy path
            $escapedHost = preg_quote($currentHost, '#');
            $escapedProxyScript = preg_quote($proxyScriptPath, '#');
            
            // Only rewrite URLs that point to current host but don't already have proxy script
            // Use negative lookahead to avoid matching URLs that already contain proxy script
            $patterns = [
                '#(https://' . $escapedHost . ')(?!' . $escapedProxyScript . ')(/[^"\'\s>]*)#i',
                '#(http://' . $escapedHost . ')(?!' . $escapedProxyScript . ')(/[^"\'\s>]*)#i',
                '#(//' . $escapedHost . ')(?!' . $escapedProxyScript . ')(/[^"\'\s>]*)#i'
            ];
            
            foreach ($patterns as $pattern) {
                $body = preg_replace($pattern, '$1' . $proxyScriptPath . '$2', $body);
            }
        }
        
        // Special handling for window.routerBase to ensure API requests use correct base path
        $currentProxyPath = parse_url($proxyUrl, PHP_URL_PATH);
        if ($currentProxyPath && strpos($body, 'window.routerBase') !== false) {
            // Ensure the proxy path ends with a slash for proper URL construction
            $routerBasePath = rtrim($currentProxyPath, '/') . '/';
            // Replace window.routerBase = "/" with the correct proxy path
            $body = preg_replace(
                '#window\.routerBase\s*=\s*["\'][^"\']*["\']#i',
                'window.routerBase = "' . $routerBasePath . '"',
                $body
            );
        }
    }
}

// Output body
error_log("Proxy: About to output body, length: " . strlen($body ?: ''));
echo $body;
error_log("Proxy: Body output completed");
exit;
?>