<?php
/**
 * 数据库鉴权功能
 * 连接到xboard数据库验证用户登录
 */

class DatabaseAuth {
    private $pdo;
    private $config;
    
    public function __construct() {
        $this->loadConfig();
        try {
            $this->connect();
        } catch (Exception $e) {
            // 连接失败时不抛出异常，让isConnected()方法处理
            error_log('Database connection failed: ' . $e->getMessage());
            $this->pdo = null;
        }
    }
    
    /**
     * 加载数据库配置
     */
    private function loadConfig() {
        $configFile = __DIR__ . '/config.json';
        if (!file_exists($configFile)) {
            throw new Exception('Database config file not found');
        }
        
        $this->config = json_decode(file_get_contents($configFile), true);
        if (!$this->config) {
            throw new Exception('Invalid database config format');
        }
    }
    
    /**
     * 连接数据库
     */
    private function connect() {
        try {
            $dsn = sprintf(
                'mysql:host=%s;port=%d;dbname=%s;charset=%s',
                $this->config['host'],
                $this->config['port'],
                $this->config['database'],
                $this->config['charset']
            );
            
            $this->pdo = new PDO($dsn, $this->config['username'], $this->config['password'], [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false
            ]);
        } catch (PDOException $e) {
            throw new Exception('Database connection failed: ' . $e->getMessage());
        }
    }
    
    /**
     * 验证用户登录
     * @param string $email 用户邮箱
     * @param string $password 用户密码
     * @return array|false 成功返回用户信息，失败返回false
     */
    public function validateUser($email, $password) {
        if (!$this->isConnected() || $this->pdo === null) {
            return false;
        }
        
        try {
            $sql = sprintf(
                'SELECT %s, %s, expired_at FROM %s WHERE %s = ? LIMIT 1',
                $this->config['email_field'],
                $this->config['password_field'],
                $this->config['table'],
                $this->config['email_field']
            );
            
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute([$email]);
            $user = $stmt->fetch();
            
            if ($user === false || !$user) {
                return false;
            }
            
            // 验证密码 - 支持多种加密方式
            if ($this->verifyPassword($password, $user[$this->config['password_field']])) {
                return [
                    'email' => $user[$this->config['email_field']],
                    'expired_at' => $user['expired_at'],
                    'authenticated' => true
                ];
            }
            
            return false;
        } catch (PDOException $e) {
            error_log('Database auth error: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * 获取用户到期时间信息
     * @param string $email 用户邮箱
     * @return array|false 成功返回到期信息，失败返回false
     */
    public function getUserExpiryInfo($email) {
        if (!$this->isConnected() || $this->pdo === null) {
            return false;
        }
        
        try {
            $sql = sprintf(
                'SELECT expired_at, remarks FROM %s WHERE %s = ? LIMIT 1',
                $this->config['table'],
                $this->config['email_field']
            );
            
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute([$email]);
            $user = $stmt->fetch();
            
            if ($user === false || !$user || !isset($user['expired_at'])) {
                return false;
            }
            
            $expiredAt = (int)$user['expired_at'];
            $currentTime = time();
            $daysDiff = floor(($expiredAt - $currentTime) / (24 * 60 * 60));
            
            // 获取用户备注信息，如果为空则使用邮箱
            $userRemarks = !empty($user['remarks']) ? $user['remarks'] : $email;
            
            if ($daysDiff > 0) {
                return [
                    'status' => 'active',
                    'days_remaining' => $daysDiff,
                    'days_number' => $daysDiff, // 单独的数字字段
                    'user_name' => $userRemarks,
                    'show_panel_button' => $daysDiff <= 5, // 剩余天数小于等于5天时显示面板按钮
                    'status_message' => "您的账户还有 {$daysDiff} 天到期",
                    'action_message' => $daysDiff <= 5 ? "感谢您的使用，如需继续使用请查看您的邮箱收件箱" : ""
                ];
            } else {
                $daysExpired = abs($daysDiff);
                return [
                    'status' => 'expired',
                    'days_expired' => $daysExpired,
                    'days_number' => $daysExpired, // 单独的数字字段
                    'user_name' => $userRemarks,
                    'show_panel_button' => true, // 已过期时显示面板按钮
                    'status_message' => "您的账户已过期 {$daysExpired} 天",
                    'action_message' => "感谢您的使用，如需继续使用请查看您的邮箱收件箱"
                ];
            }
        } catch (PDOException $e) {
            error_log('Database expiry check error: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * 验证密码
     * @param string $password 明文密码
     * @param string $hash 数据库中的密码哈希
     * @return bool
     */
    private function verifyPassword($password, $hash) {
        // 支持多种密码验证方式
        
        // 1. PHP password_verify (推荐)
        if (password_verify($password, $hash)) {
            return true;
        }
        
        // 2. MD5 (常见的旧系统)
        if (md5($password) === $hash) {
            return true;
        }
        
        // 3. SHA1
        if (sha1($password) === $hash) {
            return true;
        }
        
        // 4. SHA256
        if (hash('sha256', $password) === $hash) {
            return true;
        }
        
        // 5. 直接比较 (明文密码，不推荐但可能存在)
        if ($password === $hash) {
            return true;
        }
        
        return false;
    }
    
    /**
     * 获取数据库连接状态
     * @return bool
     */
    public function isConnected() {
        try {
            return $this->pdo && $this->pdo->query('SELECT 1')->fetchColumn() == 1;
        } catch (PDOException $e) {
            return false;
        }
    }
    


    /**
     * 测试数据库连接
     * @return array
     */
    public function testConnection() {
        try {
            $result = [
                'connected' => $this->isConnected(),
                'config' => [
                    'host' => $this->config['host'],
                    'database' => $this->config['database'],
                    'port' => $this->config['port'],
                    'table' => $this->config['table']
                ]
            ];
            
            if ($result['connected'] && $this->pdo !== null) {
                // 检查表是否存在
                $stmt = $this->pdo->query("SHOW TABLES LIKE '" . $this->config['table'] . "'");
                $result['table_exists'] = $stmt->fetch() !== false;
                
                if ($result['table_exists']) {
                    // 检查字段是否存在
                    $stmt = $this->pdo->query('DESCRIBE ' . $this->config['table']);
                    $columns = $stmt->fetchAll(PDO::FETCH_COLUMN);
                    
                    $result['email_field_exists'] = in_array($this->config['email_field'], $columns);
                    $result['password_field_exists'] = in_array($this->config['password_field'], $columns);
                }
            }
            
            return $result;
        } catch (Exception $e) {
            return [
                'connected' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * 获取用户到期日期
     * @param string $email 用户邮箱
     * @return string|null 到期日期 (Y-m-d格式) 或 null
     */
    public function getUserExpiryDate($email) {
        if (!$this->isConnected() || $this->pdo === null) {
            return null;
        }
        
        try {
            $sql = sprintf(
                'SELECT expired_at FROM %s WHERE %s = ? LIMIT 1',
                $this->config['table'],
                $this->config['email_field']
            );
            
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute([$email]);
            $user = $stmt->fetch();
            
            if ($user === false || !$user || !isset($user['expired_at'])) {
                return null;
            }
            
            // 将时间戳转换为日期格式
            return date('Y-m-d', (int)$user['expired_at']);
        } catch (PDOException $e) {
            error_log('Database error in getUserExpiryDate: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * 计算距离到期还有多少天
     * @param string $email 用户邮箱
     * @return int|null 剩余天数，负数表示已过期，null表示无法获取
     */
    public function getDaysUntilExpiry($email) {
        $expiryDate = $this->getUserExpiryDate($email);
        if (!$expiryDate) {
            return null;
        }
        
        $today = new DateTime();
        $expiry = new DateTime($expiryDate);
        $interval = $today->diff($expiry);
        
        // 如果已过期，返回负数
        if ($today > $expiry) {
            return -$interval->days;
        }
        
        return $interval->days;
    }

    /**
     * 获取用户名称（remarks字段）
     * @param string $email 用户邮箱
     * @return string|null 用户名称，null表示无法获取
     */
    public function getUserName($email) {
        if (!$this->isConnected() || $this->pdo === null) {
            return $email; // 如果数据库未连接，返回邮箱作为用户名
        }

        try {
            $stmt = $this->pdo->prepare("SELECT remarks FROM v2_user WHERE email = ?");
            $stmt->execute([$email]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($user === false || !$user) {
                return null;
            }
            
            return $user['remarks'] ?: $email; // 如果remarks为空，返回邮箱
        } catch (PDOException $e) {
            error_log('Database error in getUserName: ' . $e->getMessage());
            return $email; // 数据库错误时返回邮箱作为用户名
        }
    }
}
?>