<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);

/**
 * Simple proxy that forwards request headers (including Cookie),
 * preserves Set-Cookie in responses, rewrites Location headers
 * to keep redirects inside the proxy, and handles missing config/cURL.
 */

$configFile = __DIR__ . '/.config/auth/config.json';
if (!file_exists($configFile)) {
    http_response_code(500);
    die('Config file not found.');
}
$config = json_decode(file_get_contents($configFile), true);
$targetBase = rtrim($config['redirectUrl'] ?? '', '/');
if (!$targetBase) {
    http_response_code(500);
    die('Missing redirectUrl in config.');
}

// Build target URL by appending the request path (strip script name)
$requestUri = $_SERVER['REQUEST_URI'] ?? '/';
$scriptName = $_SERVER['SCRIPT_NAME'] ?? '/proxy.php';
$path = preg_replace('#^' . preg_quote($scriptName) . '#', '', $requestUri);

// Special handling for favicon.ico - redirect to local favicon.svg
if ($path === '/favicon.ico' || $path === 'favicon.ico') {
    header('Location: /src/favicon.svg');
    exit;
}

// Special handling for local files (config, auth system, tools, etc.)
$localFilePaths = ['/.config/', '.config/', '/auth.js', 'auth.js', '/tools/', 'tools/', '/src/', 'src/'];
foreach ($localFilePaths as $localPrefix) {
    if (strpos($path, $localPrefix) === 0) {
        $localPath = __DIR__ . $path;
        if (file_exists($localPath)) {
            $extension = pathinfo($localPath, PATHINFO_EXTENSION);
            $mimeType = 'text/plain';
            switch ($extension) {
                case 'json':
                    $mimeType = 'application/json';
                    break;
                case 'js':
                    $mimeType = 'application/javascript';
                    break;
                case 'html':
                    $mimeType = 'text/html';
                    break;
                case 'css':
                    $mimeType = 'text/css';
                    break;
                case 'svg':
                    $mimeType = 'image/svg+xml';
                    break;
                case 'png':
                    $mimeType = 'image/png';
                    break;
                case 'jpg':
                case 'jpeg':
                    $mimeType = 'image/jpeg';
                    break;
            }
            header('Content-Type: ' . $mimeType);
            readfile($localPath);
            exit;
        } else {
            http_response_code(404);
            die('Local file not found.');
        }
    }
}

$targetUrl = $targetBase . ($path === '' ? '/' : $path);

// Initialize cURL
$ch = curl_init();
if ($ch === false) {
    http_response_code(500);
    die('Failed to initialize cURL.');
}
curl_setopt($ch, CURLOPT_URL, $targetUrl);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_HEADER, true);
curl_setopt($ch, CURLOPT_FOLLOWLOCATION, false);
curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
curl_setopt($ch, CURLOPT_TIMEOUT, 60);

// Method and body
$method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);
$body = file_get_contents('php://input');
if ($body !== false && $body !== '') {
    curl_setopt($ch, CURLOPT_POSTFIELDS, $body);
}

// Forward request headers (preserve Cookie and other important headers)
$requestHeaders = [];
if (function_exists('getallheaders')) {
    foreach (getallheaders() as $name => $value) {
        $lname = strtolower($name);
        // Skip headers that should be regenerated by cURL or proxy
        if (in_array($lname, ['host', 'content-length', 'connection', 'accept-encoding'])) {
            continue;
        }
        $requestHeaders[] = $name . ': ' . $value;
    }
}

// Add User-Agent if not present
$hasUserAgent = false;
foreach ($requestHeaders as $header) {
    if (stripos($header, 'user-agent:') === 0) {
        $hasUserAgent = true;
        break;
    }
}
if (!$hasUserAgent) {
    $requestHeaders[] = 'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36';
}

// Prepare proxy URL variables for later use
$scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https://' : 'http://';
$proxyHost = $_SERVER['HTTP_HOST'] ?? 'localhost';
$proxyScript = $_SERVER['SCRIPT_NAME'] ?? '/proxy.php'; // e.g., /proxy.php
// Ensure proxyScript starts with a slash
if (substr($proxyScript, 0, 1) !== '/') {
    $proxyScript = '/' . $proxyScript;
}
$proxyUrl = $scheme . $proxyHost . $proxyScript;

// Ensure Host header matches target host
$targetHost = parse_url($targetBase, PHP_URL_HOST);
if ($targetHost) {
    $requestHeaders[] = 'Host: ' . $targetHost;
}

// Add essential headers for better compatibility
$requestHeaders[] = 'User-Agent: Mozilla/5.0 (compatible; ProxyBot/1.0)';

// Handle referer rewriting
if (isset($_SERVER['HTTP_REFERER'])) {
    $referer = $_SERVER['HTTP_REFERER'];
    // Rewrite referer to point to target domain
    $referer = str_ireplace($scheme . $proxyHost . $proxyScript, $targetBase, $referer);
    $requestHeaders[] = 'Referer: ' . $referer;
}

curl_setopt($ch, CURLOPT_HTTPHEADER, $requestHeaders);

// Execute the request
$response = curl_exec($ch);

// Check for cURL errors
if ($response === false) {
    $error = curl_error($ch);
    $errno = curl_errno($ch);
    curl_close($ch);
    http_response_code(502);
    die("cURL Error ($errno): $error");
}

$headerSize = curl_getinfo($ch, CURLINFO_HEADER_SIZE);
$status = curl_getinfo($ch, CURLINFO_HTTP_CODE) ?: 200;
curl_close($ch);

$headerText = substr($response, 0, $headerSize);
$body = substr($response, $headerSize);

// Proxy URL variables are already defined above

// Send response status
http_response_code($status);

// Parse and forward headers, and find Content-Type
$contentType = '';
$lines = preg_split("/\r\n|\n|\r/", $headerText);
foreach ($lines as $line) {
    if (trim($line) === '') continue;
    if (stripos($line, 'HTTP/') === 0) continue;

    // skip hop-by-hop headers that should be regenerated by the proxy
    if (stripos($line, 'Transfer-Encoding:') === 0 ||
        stripos($line, 'Content-Length:') === 0 ||
        stripos($line, 'Content-Encoding:') === 0 ||
        stripos($line, 'Connection:') === 0) {
        continue;
    }

    // Extract Content-Type for body rewriting later
    if (stripos($line, 'Content-Type:') === 0) {
        $contentType = trim(substr($line, strlen('Content-Type:')));
    }
    
    // Rewrite Location to stay behind proxy
    if (stripos($line, 'Location:') === 0) {
        $location = trim(substr($line, strlen('Location:')));
        // Replace target base URL with proxy script URL
        $rewritten = str_ireplace($targetBase, $proxyUrl, $location);
        header('Location: ' . $rewritten, true);
        continue;
    }

    header($line, false);
}

// Rewrite URLs in various content types
$needsRewriting = false;
if (stripos($contentType, 'text/html') !== false || 
    stripos($contentType, 'text/css') !== false || 
    stripos($contentType, 'application/javascript') !== false ||
    stripos($contentType, 'text/javascript') !== false) {
    $needsRewriting = true;
}

if ($needsRewriting) {
    // First, handle relative paths in JavaScript and CSS before absolute URL replacement
    
    // For JavaScript content, handle common URL patterns
    // Temporarily disabled to avoid conflicts with absolute URL replacement
    // if (stripos($contentType, 'javascript') !== false) {
    //     $body = preg_replace('#(?<=[\s,\[\{])(["\'])(/[^/])#', '$1' . $proxyScript . '$2', $body);
    // }
    
    // For CSS content, handle url() references
    if (stripos($contentType, 'text/css') !== false) {
        // Handle url("/path") and url('/path') in CSS
        $body = preg_replace('#url\((["\']?)/#i', 'url($1' . $proxyScript . '/', $body);
    }
    
    // For HTML content, handle attributes
    if (stripos($contentType, 'text/html') !== false && stripos($contentType, 'javascript') === false) {
        // Prepend proxy script path to root-relative URLs in common attributes
        // Only match when it's actually in an HTML tag (preceded by < and tag name)
        $body = preg_replace('#(<[^>]*\s)(href|src|action|formaction|background|data-src|data-url)=(["\'])/#i', '$1$2=$3' . $proxyScript . '/', $body);
        
        // Handle meta refresh redirects
        // Only match when it's actually in a meta tag
        $body = preg_replace('#(<meta[^>]*content=(["\'])\d+;\s*url=)/#i', '$1' . $proxyScript . '/', $body);
    }
    
    // Finally, replace absolute URLs pointing to the target with our proxy URL
    $targetHost = parse_url($targetBase, PHP_URL_HOST);
    if ($targetHost) {
        // Only replace URLs that exactly match our target base URL
        $body = str_ireplace($targetBase, $proxyUrl, $body);
        // Also handle protocol-relative URLs for the target host only
        $body = str_ireplace('//' . $targetHost, $proxyUrl, $body);
    }
    
    // Handle URLs pointing to the current proxy domain, but avoid double-rewriting
    // Only rewrite if the URL doesn't already contain the proxy script path
    $currentHost = $_SERVER['HTTP_HOST'] ?? '';
    if ($currentHost) {
        // Only rewrite URLs that don't already contain our proxy script path
        $proxyScriptPath = parse_url($proxyUrl, PHP_URL_PATH);
        
        // More precise patterns to avoid rewriting already correct proxy URLs
        // We need to ensure we don't match URLs that already contain the full proxy path
        $escapedHost = preg_quote($currentHost, '#');
        $escapedProxyScript = preg_quote($proxyScriptPath, '#');
        
        // Only rewrite URLs that point to current host but don't already have proxy script
        // Use negative lookahead to avoid matching URLs that already contain proxy script
        $patterns = [
            '#(https://' . $escapedHost . ')(?!' . $escapedProxyScript . ')(/[^"\'\s>]*)#i',
            '#(http://' . $escapedHost . ')(?!' . $escapedProxyScript . ')(/[^"\'\s>]*)#i',
            '#(//' . $escapedHost . ')(?!' . $escapedProxyScript . ')(/[^"\'\s>]*)#i'
        ];
        
        foreach ($patterns as $pattern) {
            $body = preg_replace($pattern, '$1' . $proxyScriptPath . '$2', $body);
        }
    }
    
    // Special handling for window.routerBase to ensure API requests use correct base path
    $currentProxyPath = parse_url($proxyUrl, PHP_URL_PATH);
    if ($currentProxyPath && strpos($body, 'window.routerBase') !== false) {
        // Ensure the proxy path ends with a slash for proper URL construction
        $routerBasePath = rtrim($currentProxyPath, '/') . '/';
        // Replace window.routerBase = "/" with the correct proxy path
        $body = preg_replace(
            '#window\.routerBase\s*=\s*["\'][^"\']*["\']#i',
            'window.routerBase = "' . $routerBasePath . '"',
            $body
        );
    }
}

// Output body
echo $body;
exit;
?>