// 安全认证系统
class AuthSystem {
    constructor() {
        this.config = null;
        this.maxAttempts = 3;
        this.attemptCount = 0;
        this.lockoutTime = 5 * 60 * 1000; // 5分钟锁定
        this.init();
    }

    async init() {
        try {
            // 加载配置
            const response = await fetch(`.config/auth/config.json?t=${new Date().getTime()}`);
            if (!response.ok) {
                throw new Error('配置文件加载失败');
            }
            this.config = await response.json();
            // 确保config对象包含必要的属性，即使它们是空的
            if (!this.config.passwordHash) this.config.passwordHash = '';
            if (!this.config.redirectUrl) this.config.redirectUrl = '';
            if (!this.config.salt) this.config.salt = '';
        } catch (error) {
            console.error('配置加载错误:', error);
            this.showError('系统配置错误，请联系管理员。');
            const loginBtn = document.getElementById('loginBtn');
            if(loginBtn) loginBtn.disabled = true;
        }
        
        this.bindEvents();
        this.checkLockout();
    }

    bindEvents() {
        const form = document.getElementById('loginForm');
        const passwordInput = document.getElementById('password');
        const loginBtn = document.getElementById('loginBtn');

        // 表单提交事件
        if (form) {
            form.addEventListener('submit', (e) => {
                e.preventDefault();
                this.handleLogin();
            });
        }

        // 密码输入框回车事件
        if (passwordInput) {
            passwordInput.addEventListener('keypress', (e) => {
                if (e.key === 'Enter') {
                    e.preventDefault();
                    this.handleLogin();
                }
            });
        }

        // 登录按钮点击事件
        if (loginBtn) {
            loginBtn.addEventListener('click', (e) => {
                e.preventDefault();
                this.handleLogin();
            });
        }

        // 防止右键和F12
        document.addEventListener('contextmenu', (e) => e.preventDefault());
        document.addEventListener('keydown', (e) => {
            if (e.key === 'F12' || 
                (e.ctrlKey && e.shiftKey && e.key === 'I') ||
                (e.ctrlKey && e.shiftKey && e.key === 'C') ||
                (e.ctrlKey && e.key === 'u')) {
                e.preventDefault();
                this.redirectToSpace();
            }
        });

        // 防止选择文本
        document.addEventListener('selectstart', (e) => e.preventDefault());
        document.addEventListener('dragstart', (e) => e.preventDefault());
    }

    async handleLogin() {
        if (this.isLockedOut()) {
            this.showError('登录尝试过多，请稍后再试');
            return;
        }

        const password = document.getElementById('password').value;

        if (!password) {
            this.showError('请输入密码');
            return;
        }

        this.showLoading(true);

        // 模拟网络延迟，增加安全性
        await this.delay(1000 + Math.random() * 1000);

        if (await this.validateCredentials(password)) {
            this.onLoginSuccess();
        } else {
            this.onLoginFailure();
        }

        this.showLoading(false);
    }

    async validateCredentials(password) {
        if (!this.config) return false;

        // 使用SHA-256进行密码哈希验证
        const hashedPassword = await this.hashPassword(password);

        // 只验证密码
        return hashedPassword === this.config.passwordHash;
    }

    async hashPassword(text) {
        const encoder = new TextEncoder();
        const data = encoder.encode(text + this.config.salt);
        const hashBuffer = await crypto.subtle.digest('SHA-256', data);
        const hashArray = Array.from(new Uint8Array(hashBuffer));
        return hashArray.map(b => b.toString(16).padStart(2, '0')).join('');
    }

    onLoginSuccess() {
        // 清除尝试次数
        this.attemptCount = 0;
        localStorage.removeItem('loginAttempts');
        localStorage.removeItem('lockoutTime');

        // 设置认证令牌
        const token = this.generateToken();
        sessionStorage.setItem('authToken', token);
        
        // 设置cookie供PHP检测（会话级别，浏览器关闭后失效）
        document.cookie = `authToken=${token}; path=/; SameSite=Strict`;
        
        // 跳转到根目录（现在会通过PHP路由器处理）
        window.location.href = '/';
    }

    onLoginFailure() {
        // 记录鉴权失败信息
        this.recordFailedAttempt(document.getElementById('password').value);

        // 清空输入框
        document.getElementById('password').value = '';
        
        // 鉴权失败时，使用DOMParser在当前页面显示503内容，URL不变
        fetch('503.html')
            .then(response => response.text())
            .then(html => {
                const parser = new DOMParser();
                const doc = parser.parseFromString(html, 'text/html');

                // 替换 head 和 body
                document.head.innerHTML = doc.head.innerHTML;
                document.body.innerHTML = doc.body.innerHTML;

                // 重新执行 503.html 中的脚本以确保跳转功能生效
                const scripts = Array.from(doc.body.getElementsByTagName("script"));
                scripts.forEach(oldScript => {
                    const newScript = document.createElement("script");
                    if (oldScript.src) {
                        newScript.src = oldScript.src;
                    } else {
                        newScript.textContent = oldScript.textContent;
                    }
                    document.body.appendChild(newScript);
                });
            });
    }

    recordFailedAttempt(passwordAttempt) {
        // 新增：发送失败日志到服务端
        fetch('tools/log_failed_attempt.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                timestamp: Date.now(),
                passwordAttempt: passwordAttempt
            })
        }).catch(e => { console.error('Failed to log attempt:', e); });

        // 保留本地记录（可选）
        let failedAttempts = [];
        try {
            const storedAttempts = localStorage.getItem('failedAuthAttempts');
            if (storedAttempts) {
                failedAttempts = JSON.parse(storedAttempts);
            }
        } catch (e) {
            console.error("Error reading failed attempts from localStorage:", e);
        }

        const newAttempt = {
            timestamp: Date.now(),
            ip: '未知',
            passwordAttempt: passwordAttempt
        };
        failedAttempts.push(newAttempt);
        localStorage.setItem('failedAuthAttempts', JSON.stringify(failedAttempts));
    }

    generateToken() {
        const timestamp = Date.now();
        const random = Math.random().toString(36).substr(2, 9);
        return btoa(`${timestamp}-${random}`);
    }

    checkLockout() {
        const attempts = localStorage.getItem('loginAttempts');
        const lockoutTime = localStorage.getItem('lockoutTime');
        
        if (attempts) {
            this.attemptCount = parseInt(attempts);
        }
        
        if (lockoutTime && this.isLockedOut()) {
            const remainingTime = Math.ceil((parseInt(lockoutTime) + this.lockoutTime - Date.now()) / 1000 / 60);
            this.showError(`账户已锁定，请${remainingTime}分钟后再试`);
            document.getElementById('loginForm').style.display = 'none';
        }
    }

    isLockedOut() {
        const lockoutTime = localStorage.getItem('lockoutTime');
        if (!lockoutTime) return false;
        
        return Date.now() - parseInt(lockoutTime) < this.lockoutTime;
    }

    showError(message) {
        const errorDiv = document.getElementById('errorMessage');
        errorDiv.textContent = message;
        errorDiv.style.display = 'block';
        
        // 3秒后自动隐藏错误信息
        setTimeout(() => {
            errorDiv.style.display = 'none';
        }, 3000);
    }

    showLoading(show) {
        const loading = document.getElementById('loading');
        const button = document.querySelector('.login-btn');
        
        if (show) {
            loading.style.display = 'block';
            button.disabled = true;
            button.textContent = '验证中...';
        } else {
            loading.style.display = 'none';
            button.disabled = false;
            button.textContent = '登录';
        }
    }

    redirectToSpace() {
        window.location.href = '503.html';
    }

    delay(ms) {
        return new Promise(resolve => setTimeout(resolve, ms));
    }
}

// 页面加载完成后初始化认证系统
document.addEventListener('DOMContentLoaded', () => {
    new AuthSystem();
});

// 防止通过控制台访问敏感信息
(function() {
    let devtools = {
        open: false,
        orientation: null
    };
    
    const threshold = 160;
    
    setInterval(() => {
        if (window.outerHeight - window.innerHeight > threshold || 
            window.outerWidth - window.innerWidth > threshold) {
            if (!devtools.open) {
                devtools.open = true;
                // 仅标记开发者工具已打开，不进行跳转
            }
        } else {
            devtools.open = false;
        }
    }, 500);
})();

// 清理控制台
console.clear();
console.log('%c🚀 欢迎来到外太空！', 'color: #667eea; font-size: 20px; font-weight: bold;');
console.log('%c如果你看到这条消息，说明你已经偏离了正常轨道...', 'color: #764ba2; font-size: 14px;');